#ifndef _YIMAP_PARSER_SEQUENCE_ITERATOR_H_
#define _YIMAP_PARSER_SEQUENCE_ITERATOR_H_

#include <common/spirit_defs.h>
#include <common/tree.h>
#include <common/lex_ids.h>

#include <boost/iterator/iterator_facade.hpp>
#include <boost/type_traits/is_convertible.hpp>
#include <boost/utility/enable_if.hpp>

namespace yimap { namespace parser {

template <typename Value>
class SequenceIterator
    : public boost::iterator_facade<SequenceIterator<Value>, Value, boost::forward_traversal_tag>
{
private:
    struct enabler
    {
    };

public:
    SequenceIterator(void) : top_(0)
    {
    }
    SequenceIterator(const TreeNode& __node);

    template <typename OtherValue>
    SequenceIterator(
        const SequenceIterator<OtherValue>& other,
        typename boost::enable_if<boost::is_convertible<OtherValue*, Value*>, enabler>::type =
            enabler())
        : top_(other.top_)
        , iter_(other.iter_)
        , number_(other.number_)
        , lnumber_(other.lnumber_)
        , range_mode_(other.range_mode_)
    {
    }

private:
    friend class boost::iterator_core_access;

    template <typename OtherValue>
    bool equal(
        const SequenceIterator<OtherValue>& other,
        typename boost::enable_if<boost::is_convertible<OtherValue*, Value*>, enabler>::type =
            enabler()) const;

    void increment(void);
    Value& dereference() const;

protected:
    void generate_number(void);

private:
    const TreeNode* top_;
    TreeNode::const_tree_iterator iter_;
    unsigned int number_;
    unsigned int lnumber_;
    bool range_mode_;
};

template <typename Value>
SequenceIterator<Value>::SequenceIterator(const TreeNode& __node)
    : top_(&__node), iter_(__node.children.begin())
{
    const TreeNode::parse_node_t& val = top_->value;
    assert(val.id().to_long() == lex_ids::SEQ_SET);

    if (this->iter_ == this->top_->children.end())
    {
        this->top_ = 0;
        return;
    }

    this->generate_number();
}

template <typename Value>
void SequenceIterator<Value>::generate_number(void)
{
    assert(this->top_ != 0);
    const TreeNode::parse_node_t& val = this->iter_->value;
    if (val.id().to_long() == lex_ids::UNSIGNED)
    {
        this->number_ = boost::get<unsigned int>(val.value());
        this->range_mode_ = false;
    }
    else
    {
        TreeNode::const_tree_iterator i = this->iter_->children.begin();
        this->number_ = boost::get<unsigned int>(i->value.value());
        this->lnumber_ = boost::get<unsigned int>((++i)->value.value());

        if (this->number_ > this->lnumber_) throw std::runtime_error("bad sequence");

        this->range_mode_ = true;
    }
}

template <typename Value>
void SequenceIterator<Value>::increment(void)
{
    assert(this->top_ != 0);
    if (this->range_mode_ && ++this->number_ <= this->lnumber_) return;

    if (++this->iter_ == this->top_->children.end()) this->top_ = 0;
    else
        this->generate_number();
}

template <typename Value>
template <typename OtherValue>
bool SequenceIterator<Value>::equal(
    const SequenceIterator<OtherValue>& other,
    typename boost::enable_if<boost::is_convertible<OtherValue*, Value*>, enabler>::type) const
{
    if (this->top_ != other.top_) return false;
    if (this->top_ == 0) return true;

    return (
        this->iter_ == other.iter_ && this->number_ == other.number_ &&
        this->range_mode_ == other.range_mode_ && this->lnumber_ == other.lnumber_);
}

template <typename Value>
Value& SequenceIterator<Value>::dereference(void) const
{
    assert(this->top_ != 0);
    return this->number_;
}

typedef SequenceIterator<unsigned int> seq_iterator;
typedef SequenceIterator<const unsigned int> seq_const_iterator;

}}
#endif // _YIMAP_PARSER_SEQUENCE_ITERATOR_H_
