#pragma once

#include <commands/imap_command.h>
#include <commands/factory.h>
#include <common/types.h>
#include <backend/backend.h>
#include <network/session.h>
#include <parser/preparser.h>
#include <parser/parser.h>
#include <typed_log/typed_log.h>
#include <yplatform/find.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace yimap {

using Coroutine = boost::asio::coroutine;
#define coro_exit yield break

struct ReadAndExecCommandsLoop
{
    using YieldCtx = yplatform::yield_context<ReadAndExecCommandsLoop>;
    using Preparser = yimap::Preparser<zerocopy::BufferIt>;

    ImapContextPtr context;
    NetworkSessionPtr session;
    SettingsCPtr settings;
    backend::BundlePtr backends;
    IOService& ioService;
    SessionLogger& logger;
    Preparser preparser;
    MessagesStats messagesStatsSnapshot;

    // String like ". sc=LZivkqb04mI1" for error output ending.
    const string errorSuffix;

    PreparserResult preparsed;
    zerocopy::Segment commandText;
    CommandPtr command;

    ReadAndExecCommandsLoop(
        ImapContextPtr context,
        NetworkSessionPtr session,
        backend::BundlePtr backends)
        : context(context)
        , session(session)
        , settings(context->settings)
        , backends(backends)
        , ioService(context->ioService)
        , logger(context->sessionLogger)
        , errorSuffix(createErrorSuffix(context->uniq_id()))
    {
    }

    void operator()(YieldCtx yieldCtx, const ErrorCode& ec = {})
    {
        // TODO add coro ref to ctx for debug
        // TODO try-catch
        reenter(yieldCtx)
        {
            while (true)
            {
                yield asyncReadData(yieldCtx);

                if (ec)
                {
                    if (autoLogout(ec))
                    {
                        sendByeIdleAutoLogout();
                    }
                    coro_exit;
                }

                while (true)
                {
                    preparsed = preparseCommand();

                    if (!preparsed.commandComplete && commandLimitsExceeded(preparsed))
                    {
                        commandText = consumeAllData();
                        logCommand(commandText);
                        logCommandTooLarge();
                        sendByeTooLarge();
                        coro_exit;
                    }

                    if (preparsed.continuationRequests > 0)
                    {
                        sendContinuationRequests(preparsed.continuationRequests);
                    }

                    if (!preparsed.commandComplete)
                    {
                        break; // read more
                    }

                    commandText = consumeData(preparsed.totalSize);
                    resetPreparser();

                    if (commandLimitsExceeded(preparsed))
                    {
                        sendNoTooLarge(preparsed.commandTag);
                        continue;
                    }

                    assert(preparsed.commandComplete);
                    logCommand(commandText);

                    yield asyncParseCommand(yieldCtx);
                    if (ec)
                    {
                        sendBadInternalError(preparsed.commandTag);
                        coro_exit;
                    }
                    if (!command)
                    {
                        reportBadCommand();
                        sendBadSyntaxError(preparsed.commandTag);
                        continue; // skip command
                    }

                    yield asyncExecCommand(yieldCtx);

                    if (sessionLostOrClosedByCommand())
                    {
                        coro_exit;
                    }

                    if (!tooManyBadCommands())
                    {
                        sendByeTooManyCommands();
                        coro_exit;
                    }
                }
            }
        }

        if (yieldCtx.is_complete())
        {
            session->shutdown();
        }
    }

    void asyncReadData(YieldCtx yieldCtx)
    {
        session->asyncRead(1, yieldCtx);
    }

    bool autoLogout(const ErrorCode& ec)
    {
        return ec == boost::asio::error::operation_aborted && session->isOpen();
    }

    void asyncParseCommand(YieldCtx yieldCtx)
    {
        command = {};
        asyncParseCommandAST(yieldCtx, [this, yieldCtx](auto&& commandAst) {
            if (!commandAst)
            {
                yieldCtx(boost::asio::error::shut_down); // XXX use own errc
                return;
            }
            if (commandAst->data.full)
            {
                command = createCommandFromAst(commandAst);
            }
            yieldCtx();
        });
    }

    template <typename Handler>
    void asyncParseCommandAST(YieldCtx yieldCtx, Handler handler)
    {
        auto future =
            yplatform::find<parser::parser>("imap_parser")
                ->parse(
                    context, boost::make_iterator_range(commandText.begin(), commandText.end()));
        future.add_callback(
            ioService.wrap([this, yieldCtx, future, handler = std::move(handler)]() {
                try
                {
                    handler(future.get());
                }
                catch (const std::exception& e)
                {
                    logger.logError() << "asyncParseCommand exception=" << e.what();
                }
            }));
    }

    CommandPtr createCommandFromAst(CommandASTPtr commandAst)
    {
        CommandPtr command;
        ImapCommandArgs commandArgs = { context,
                                        session,
                                        backends->authBackend,
                                        backends->metaBackend,
                                        backends->mbodyBackend,
                                        backends->settingsBackend,
                                        backends->notificationsBackend,
                                        backends->search,
                                        backends->append,
                                        backends->journal,
                                        settings,
                                        &logger,
                                        commandAst };
        command = createImapCommand(commandAst, commandArgs);
        command->init(context->stats);
        return command;
    }

    PreparserResult preparseCommand()
    {
        return preparser(session->readBuffer());
    }

    bool commandLimitsExceeded(const PreparserResult& command)
    {
        auto limits = settings->serverSettings.limits;
        if (command.maxLiteralSize > limits.max_literal_size) return true;
        if (command.allLiteralsSize > limits.max_all_literal_size) return true;
        if (command.pureSize > limits.max_pure_size) return true;
        return false;
    }

    zerocopy::Segment consumeData(size_t length)
    {
        return session->consumeReadBuffer(length);
    }

    zerocopy::Segment consumeAllData()
    {
        return session->consumeEntireReadBuffer();
    }

    void resetPreparser()
    {
        preparser = Preparser();
    }

    void asyncExecCommand(YieldCtx yieldCtx)
    {
        messagesStatsSnapshot = makeMessagesStatsSnapshot();
        session->disableTimeouts();

        // Use weak_ptr to avoid circular dependencies.
        // Don't wrap with in ioService to prevent loosing the shared pointer.
        command->getFuture()->add_callback(
            [ioService = &ioService, yieldCtx, weakCommand = WeakCommandPtr(command)]() {
                ioService->post(std::bind(yieldCtx, weakCommand.lock()));
            });
        calcStats(command);
        ioService.post(std::bind(&ImapCommand::start, command));
        command = {}; // command holds yieldCtx
    }

    bool sessionLostOrClosedByCommand()
    {
        return !session->isOpen();
    }

    // Handle command is executed.
    void operator()(YieldCtx yieldCtx, CommandPtr command)
    {
        if (!command)
        {
            logger.logError() << "command execute error=command destroyed";
            session->shutdown();
            return;
        }
        session->setDefaultTimeouts();
        logTiming(command);
        try
        {
            ReturnCode ret = command->getFuture()->get();
            if (ret == ReturnCode::RET_CLOSE)
            {
                // Do nothing becuase of calling shutdown by a command,
                // TODO deprecate ret codes
            }
        }
        catch (const std::exception& e)
        {
            logger.logError() << "command execute exception=" << e.what();
            sendBadInternalError(command->tag());
        }
        catch (...)
        {
            logger.logError() << "command execute exception=unknown";
            sendBadInternalError(command->tag());
        }

        (*this)(yieldCtx);
    }

    bool tooManyBadCommands() const
    {
        return !settings->maxBadCommands ||
            context->processingLoopState.badCommands < settings->maxBadCommands;
    }

    void reportBadCommand()
    {
        context->processingLoopState.badCommands++;
    }

    void sendByeTooLarge()
    {
        session->clientStream() << "* BYE command or literal size is too large\r\n";
    }

    void sendContinuationRequests(size_t count)
    {
        for (std::size_t i = 0; i < count; ++i)
        {
            session->clientStream() << "+ Ready for additional command text\r\n";
        }
    }

    void sendByeIdleAutoLogout()
    {
        logger.logEvent() << "autologout"; // TODO extract
        session->clientStream() << "* BYE Autologout; idle for too long\r\n";
    }

    void sendByeTooManyCommands()
    {
        session->clientStream() << "* BYE Too many bad commands from your client.\r\n";
    }

    void sendNoTooLarge(const string& tag)
    {
        session->clientStream() << tag << " NO command or literal size is too large\r\n";
    }

    void sendBadInternalError(const string& tag)
    {
        session->clientStream() << tag << " BAD Internal server error." << errorSuffix << "\r\n";
    }

    void sendBadSyntaxError(const string& tag)
    {
        session->clientStream() << tag << " BAD Command syntax error." << errorSuffix << "\r\n";
    }

    void logCommand(const zerocopy::SegmentRange& commandText)
    {
        logger.logCommand(commandText);
    }

    void logCommandTooLarge()
    {
        ostringstream message;
        message.setf(ios_base::boolalpha);
        message << "command too large {";
        {
            message << "commandComplete:" << preparsed.commandComplete;
            const auto& limits = settings->serverSettings.limits;
            message << ", maxLiteralSize:" << preparsed.maxLiteralSize << "/"
                    << limits.max_literal_size;
            message << ", allLiteralsSize:" << preparsed.allLiteralsSize << "/"
                    << limits.max_all_literal_size;
            message << ", pureSize:" << preparsed.pureSize << "/" << limits.max_pure_size;
        }
        message << "}";
        logger.logString(message.str());
    }

    void logTiming(CommandPtr command)
    {
        auto future = *command->getFuture();
        assert(future.ready());

        typed::logCommandComplete(
            context,
            future.has_exception() ? "exception"s : toString(future.get()),
            command->command(),
            command->commandDump(),
            preparsed.totalSize,
            command->totalTime(),
            calcStatsDiff(messagesStatsSnapshot, makeMessagesStatsSnapshot()),
            command->statExtra()); // TODO pass exception::what()

        ostringstream debugStream;
        debugStream << "command=\"" << command->commandDump() << "\" " << command->stat();
        logger.logTiming(debugStream.str());
    }

    void calcStats(CommandPtr command)
    {
        context->calcStats(command->command());
    }

    MessagesStats makeMessagesStatsSnapshot()
    {
        return context->messagesStats;
    }

    MessagesStats calcStatsDiff(const MessagesStats& oldStats, const MessagesStats& newStats)
    {
        MessagesStats res;
        res.loadedFromMetabackendCount =
            newStats.loadedFromMetabackendCount - oldStats.loadedFromMetabackendCount;
        res.loadedFromMdbCount = newStats.loadedFromMdbCount - oldStats.loadedFromMdbCount;
        res.mbodyCacheHitCount = newStats.mbodyCacheHitCount - oldStats.mbodyCacheHitCount;
        res.storageRequestsCount = newStats.storageRequestsCount - oldStats.storageRequestsCount;
        return res;
    }
};

}

#include <yplatform/unyield.h>
