package ru.yandex.autotests.innerpochta.imap.copy;

import java.util.Collection;

import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.innerpochta.imap.base.BaseTest;
import ru.yandex.autotests.innerpochta.imap.consts.base.ImapCmd;
import ru.yandex.autotests.innerpochta.imap.consts.base.MyStories;
import ru.yandex.autotests.innerpochta.imap.core.imap.ImapClient;
import ru.yandex.autotests.innerpochta.imap.responses.CopyResponse;
import ru.yandex.autotests.innerpochta.imap.structures.FolderContainer;
import ru.yandex.autotests.innerpochta.imap.utils.Utils;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.allure.annotations.Title;

import static ru.yandex.autotests.innerpochta.imap.data.TestData.allKindsOfFolders;
import static ru.yandex.autotests.innerpochta.imap.requests.CopyRequest.copy;
import static ru.yandex.autotests.innerpochta.imap.requests.CreateRequest.create;
import static ru.yandex.autotests.innerpochta.imap.requests.SelectRequest.select;
import static ru.yandex.autotests.innerpochta.imap.rules.CleanRule.withCleanBefore;

/**
 * Created with IntelliJ IDEA.
 * User: vicdev
 * Date: 06.05.14
 * Time: 19:01
 * <p/>
 * [MAILPROTO-2137]
 */
@Aqua.Test
@Title("Команда COPY. Копируем письма из различных папок в различные папки")
@Features({ImapCmd.COPY, "UID COPY"})
@Stories(MyStories.USER_FOLDERS)
@Description("Копируем различные пачки писем из одной папки\n" +
        "Позитивное тестирование")
@RunWith(Parameterized.class)
public class CopyUserFolders extends BaseTest {
    private static Class<?> currentClass = CopyUserFolders.class;

    private static final int NUMBER_OF_MESSAGE = 2;
    @ClassRule
    public static ImapClient imap = newLoginedClient(currentClass);
    @Rule
    public ImapClient prodImap = withCleanBefore(newLoginedClient(currentClass));
    private String toFolder;

    public CopyUserFolders(String toFolder) {
        this.toFolder = toFolder;
    }

    @Parameterized.Parameters(name = "userFolder - {0}")
    public static Collection<Object[]> foldersForCreate() {
        return allKindsOfFolders();
    }

    @Description("Копируем по UID из INBOX в пользовательскую папку")
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("112")
    public void uidCopyToUserFolderTest() throws Exception {
        prodImap.append().appendRandomMessagesInInbox(NUMBER_OF_MESSAGE);
        imap.select().waitMsgsInInbox(NUMBER_OF_MESSAGE);

        prodImap.request(create(toFolder)).shouldBeOk();
        imap.list().shouldSeeFolder(toFolder);

        imap.select().inbox();
        imap.request(copy(imap.search().uidAllMessages(), toFolder).uid(true)).shouldBeOk();
        imap.select().waitMsgs(toFolder, NUMBER_OF_MESSAGE);
    }


    @Description("Пытаемся скопировать письма из пустой папки\n" +
            "[MAILPROTO-2137] Ожидаемый результат: NO")
    @Stories(MyStories.JIRA)
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("114")
    public void copyWithEmptyMailboxShouldSeeNO() {
        prodImap.request(create(toFolder)).shouldBeOk();
        imap.list().shouldSeeFolder(toFolder);
        prodImap.select().waitMsgs(toFolder, 0);

        imap.request(select(toFolder)).repeatUntilOk(imap);
        imap.request(copy("1:" + NUMBER_OF_MESSAGE, toFolder)).shouldBeNo()
                .statusLineContains(CopyResponse.NO_MESSAGES);
    }

    @Description("Копируем письма в несуществующую папку\n" +
            "[MAILPROTO-2137] Ожидаемый результат: NO")
    @Stories(MyStories.JIRA)
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("113")
    public void copyInNonExistFolderShouldSeeNo() throws Exception {
        String folderName = Utils.generateName();
        prodImap.request(create(folderName)).shouldBeOk();
        imap.list().shouldSeeFolder(folderName);
        prodImap.append().appendRandomMessages(folderName, NUMBER_OF_MESSAGE);
        imap.select().waitMsgs(folderName, NUMBER_OF_MESSAGE);

        imap.request(select(folderName)).repeatUntilOk(imap);
        imap.request(copy("1:" + NUMBER_OF_MESSAGE, toFolder)).shouldBeNo()
                .statusLineContains(CopyResponse.NO_SUCH_FOLDER);
        imap.select().waitMsgs(folderName, NUMBER_OF_MESSAGE);
    }

    @Description("Копируем письма из папки в эту же папку [MAILPROTO-2137]")
    @Stories(MyStories.JIRA)
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("111")
    public void copyToThisSameUserFolder() throws Exception {
        prodImap.request(create(toFolder)).shouldBeOk();
        imap.list().shouldSeeFolder(toFolder);
        prodImap.append().appendRandomMessages(toFolder, NUMBER_OF_MESSAGE);
        imap.select().waitMsgs(toFolder, NUMBER_OF_MESSAGE);

        imap.request(select(toFolder)).repeatUntilOk(imap);
        imap.request(copy("1:" + NUMBER_OF_MESSAGE, toFolder)).shouldBeOk();
        prodImap.select().waitMsgs(toFolder, NUMBER_OF_MESSAGE * 2);
    }

    @Description("Копируем письма из папки в другую пустую папку")
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("109")
    public void copyFromUserFolderToEmptyFolder() throws Exception {
        String folderFrom = Utils.generateName();
        prodImap.request(create(folderFrom)).shouldBeOk();
        imap.list().shouldSeeFolder(folderFrom);
        prodImap.append().appendRandomMessages(folderFrom, NUMBER_OF_MESSAGE);
        imap.select().waitMsgs(folderFrom, NUMBER_OF_MESSAGE);

        prodImap.request(create(toFolder)).shouldBeOk();
        imap.list().shouldSeeFolder(toFolder);

        imap.request(select(folderFrom)).repeatUntilOk(imap);
        imap.request(copy("1:" + NUMBER_OF_MESSAGE, toFolder)).shouldBeOk();
        prodImap.select().waitMsgs(toFolder, NUMBER_OF_MESSAGE);
        //todo: хорошо бы проверить что сообщения те
    }

    @Test
    @Description("Копируем несколько сообщений из папки в подпапку\n[MAILPROTO-2137]")
    @Stories(MyStories.JIRA)
    @ru.yandex.qatools.allure.annotations.TestCaseId("110")
    public void copyFromUserFolderToEmptySubfolder() throws Exception {
        String folderFrom = Utils.generateName();
        prodImap.request(create(folderFrom)).shouldBeOk();
        imap.list().shouldSeeFolder(folderFrom);
        prodImap.append().appendRandomMessages(folderFrom, NUMBER_OF_MESSAGE);
        imap.select().waitMsgs(folderFrom, NUMBER_OF_MESSAGE);

        FolderContainer folderContainer = FolderContainer.newFolder(Utils.generateName(), toFolder);
        prodImap.request(create(folderContainer.fullName())).shouldBeOk();
        imap.list().shouldSeeFolders(folderContainer.foldersTreeAsList());

        imap.request(select(folderFrom)).repeatUntilOk(imap);
        imap.request(copy("1:" + NUMBER_OF_MESSAGE, folderContainer.fullName())).shouldBeOk();
        prodImap.select().waitMsgs(folderContainer.fullName(), NUMBER_OF_MESSAGE);
        //todo: хорошо бы проверить что сообщения те
    }

    //делаем все в одну сессию, без прода

}
