#pragma once

#include <imap_protocol/result.h>

#include <boost/algorithm/string.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/regex.hpp>
#include <iterator>
#include <string>

namespace imap_protocol::detail {

using namespace std::literals;

template <typename Iterator>
class PreparserImpl
{
    using Buffer = boost::iterator_range<Iterator>;

public:
    PreparserResult operator()(const Buffer& buffer)
    {
        start = buffer.begin();
        saved = start + processedBytes;
        end = buffer.end();

        PreparserResult res;
        do
        {
            if (state == State::ParsingLiteral)
            {
                size_t recievedSize = std::distance(saved, end);
                if (recievedSize < currentLiteralSize)
                {
                    res.commandComplete = false;
                    break;
                }

                consume(currentLiteralSize);
                state = State::ParsingCommand;
            }
            else
            {
                res.commandComplete = findEnding();
                if (!res.commandComplete)
                {
                    break;
                }

                res.commandComplete = checkLiteral();
            }
        } while (state != State::Idle);

        res.commandTag = extractTag();
        res.totalSize = processedBytes;
        res.pureSize = commandSize;
        res.allLiteralsSize = allLiteralsSize;
        res.maxLiteralSize = maxLiteralSize;
        res.continuationRequests = requiredPluses;
        requiredPluses = 0;
        return res;
    }

private:
    bool findEnding()
    {
        state = State::ParsingCommand;
        auto searchRange = boost::make_iterator_range(saved, end);
        auto endingRange = boost::algorithm::find_first(searchRange, commandEnding());

        if (endingRange.empty())
        {
            return false;
        }
        else
        {
            auto size = std::distance(saved, endingRange.end());
            consume(size);
            commandSize += size;
        }
        return true;
    }

    bool checkLiteral()
    {
        boost::match_results<Iterator> literalMatch;
        if (!boost::regex_search(
                start,
                saved,
                literalMatch,
                literalRegex(),
                boost::regex_constants::match_single_line))
        {
            state = State::Idle;
            return true;
        }

        state = State::ParsingLiteral;
        currentLiteralSize = boost::lexical_cast<std::size_t>(literalMatch[1].str());
        allLiteralsSize += currentLiteralSize;
        maxLiteralSize = std::max(maxLiteralSize, currentLiteralSize);

        bool syncLiteral = !literalMatch[2].length();
        if (syncLiteral)
        {
            ++requiredPluses;
        }
        return false;
    }

    std::string extractTag()
    {
        boost::match_results<Iterator> tagMatch;
        if (boost::regex_search(start, saved, tagMatch, tagRegex()))
        {
            return tagMatch[1].str();
        }
        return ""s;
    }

    static const boost::regex& literalRegex()
    {
        // ~ means binary data encoding, we dont care here
        // + means no additinal message from server required before data sending
        // Number in brackets is bytes count
        static const boost::regex regex{ "\\{(\\d+)(\\+?)\\}\r\n$" };
        return regex;
    }

    static const boost::regex& tagRegex()
    {
        static const boost::regex regex{ "^( ?\\S*)\\s" };
        return regex;
    }

    static const std::string& commandEnding()
    {
        static const std::string ending = "\r\n"s;
        return ending;
    }

    void consume(size_t bytes)
    {
        saved += bytes;
        processedBytes += bytes;
    }

    enum class State
    {
        Idle = 0,
        ParsingCommand,
        ParsingLiteral
    };

    State state = State::Idle;
    Iterator start;
    Iterator end;
    Iterator saved;
    std::size_t processedBytes = 0;
    std::size_t requiredPluses = 0;
    std::size_t commandSize = 0;
    std::size_t allLiteralsSize = 0;
    std::size_t currentLiteralSize = 0;
    std::size_t maxLiteralSize = 0;
    std::string commandTag;
};

}
