package ru.yandex.personal.mailimport.commons;


import java.util.function.Consumer;
import java.util.function.Function;

import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Allows to executed actions which are synchronized on equals\hashcode contract.
 * <p>
 * User: terry
 * Date: 18.11.11
 */
public class ResourceLock<T> {
    private static final Logger LOG = LoggerFactory.getLogger(ResourceLock.class);

    // use bimap to allow keep key/value pair unique
    private final BiMap<T, Entry> lockedResources = HashBiMap.create();

    public void runInLock(T resource, Consumer<T> action) {
        Entry lockEntry = lock(resource);
        try {
            action.accept(resource);
        } finally {
            unlock(lockEntry);
        }
    }

    public boolean tryRunInLock(T resource, Consumer<T> action) {
        Entry lockEntry = tryLock(resource);
        try {
            if (lockEntry != null) {
                action.accept(resource);
                return true;
            } else {
                return false;
            }
        } finally {
            if (lockEntry != null) {
                unlock(lockEntry);
            }
        }
    }

    public <R> R runInLock(T resource, Function<T, R> computable) {
        Entry lockEntry = lock(resource);
        try {
            return computable.apply(resource);
        } finally {
            unlock(lockEntry);
        }
    }

    private boolean isLocked(T resource) {
        synchronized (lockedResources) {
            return lockedResources.containsKey(resource) && lockedResources.get(resource).thread != Thread
                    .currentThread();
        }
    }

    private Entry addLock(T resource) {
        synchronized (lockedResources) {
            Entry entry = lockedResources.get(resource);
            if (entry == null) {
                lockedResources.put(resource, entry = new Entry(Thread.currentThread()));
            }
            entry.nestingDeep++;
            return entry;
        }
    }

    private Entry lock(T resource) {
        synchronized (lockedResources) {
            while (true) {
                if (isLocked(resource)) {
                    try {
                        lockedResources.wait();
                    } catch (InterruptedException e) {
                        LOG.error("Lock waiting interrupted", e);
                    }
                } else {
                    return addLock(resource);
                }
            }
        }
    }

    private Entry tryLock(T resource) {
        synchronized (lockedResources) {
            if (isLocked(resource)) {
                return null;
            } else {
                return addLock(resource);
            }
        }
    }

    private void unlock(Entry entry) {
        synchronized (lockedResources) {
            entry.nestingDeep--;
            if (entry.nestingDeep == 0) {
                // it is required to remove by value because client code may change resource object
                lockedResources.inverse().remove(entry);
                lockedResources.notifyAll();
            }
        }
    }

    private static class Entry {
        Thread thread;
        int nestingDeep;

        Entry(final Thread thread) {
            this.thread = thread;
            nestingDeep = 0;
        }
    }
}
