package ru.yandex.personal.mailimport.controllers;

import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.personal.mailimport.commons.ResourceLock;
import ru.yandex.personal.mailimport.model.MailAccount;
import ru.yandex.personal.mailimport.model.MailFolder;
import ru.yandex.personal.mailimport.model.Mailbox;
import ru.yandex.personal.mailimport.service.BackgroundMailImportTaskMessage;
import ru.yandex.personal.mailimport.service.BackgroundMailboxer;
import ru.yandex.personal.mailimport.service.ImapConnection;
import ru.yandex.personal.mailimport.service.ImapMailbox;

@Service("mailController")
public class MailController implements MailMappings {
    private final BackgroundMailboxer mailboxer;
    private final ResourceLock<MailAccount> resourceLock = new ResourceLock<>();

    @Autowired
    public MailController(BackgroundMailboxer mailboxer) {
        this.mailboxer = mailboxer;
    }

    // Blocking because of https://github.com/wildfly-extras/wildfly-camel/issues/2447
    public void synchronizeMailBox(Mailbox mailbox) {
        resourceLock.runInLock(mailbox.getMailAccount(), (account) -> {
            ImapConnection connection = new ImapConnection(account);
            ImapMailbox imapMailbox = new ImapMailbox(connection);
            imapMailbox.clear();

            for (MailFolder mf : mailbox.getFolders()) {
                imapMailbox.addFolder(mf);
            }
        });
    }

    public BackgroundMailImportTaskMessage synchronizeMailBoxAsynchronously(Mailbox mailbox) {
        return mailboxer.submitMailbox(mailbox);
    }

    public BackgroundMailImportTaskMessage getMessage(String id) {
        return mailboxer.getMessage(id);
    }

    public BackgroundMailImportTaskMessage waitForImport(String id) {
        return mailboxer.waitForCalculation(id);
    }

    public List<String> listFolders(MailAccount mailAccount) {
        ImapConnection connection = new ImapConnection(mailAccount);
        ImapMailbox imapMailbox = new ImapMailbox(connection);
        return imapMailbox.listFolders();
    }
}
