package ru.yandex.personal.mailimport.service;

import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.springframework.stereotype.Service;

import ru.yandex.personal.mailimport.model.Mailbox;

@Service
public class BackgroundMailboxer {
    private final ExecutorService executor = Executors.newCachedThreadPool();

    private final Cache<String, BackgroundMailImportTask> tasks = CacheBuilder.newBuilder()
            .expireAfterAccess(1, TimeUnit.DAYS)
            .build();

    private final Map<BackgroundMailImportTask, Future<?>> futures = new ConcurrentHashMap<>();

    public BackgroundMailImportTaskMessage submitMailbox(Mailbox mailbox) {
        BackgroundMailImportTask task = new BackgroundMailImportTask(mailbox);
        Future<?> future = executor.submit(() -> {
            task.run();
            futures.remove(task);
        });
        futures.put(task, future);

        String id = Integer.toHexString(future.hashCode()) + Integer.toHexString(mailbox.hashCode());
        tasks.put(id, task);

        return new BackgroundMailImportTaskMessage(id, task.getMessage());
    }

    public BackgroundMailImportTaskMessage getMessage(String id) {
        BackgroundMailImportTask task = Optional.ofNullable(tasks.getIfPresent(id))
                .orElseThrow(() -> new IllegalStateException("No such id: " + id));

        return new BackgroundMailImportTaskMessage(id, task.getMessage());
    }

    public BackgroundMailImportTaskMessage waitForCalculation(String id) {
        BackgroundMailImportTask task = Optional.ofNullable(tasks.getIfPresent(id))
                .orElseThrow(() -> new IllegalStateException("No such id: " + id));

        Future<?> f = futures.get(task);
        if (f != null) {
            try {
                f.get();
                return new BackgroundMailImportTaskMessage(id, task.getMessage());
            } catch (InterruptedException | ExecutionException e) {
                throw new RuntimeException(e);
            }
        } else {
            throw new IllegalStateException("No such id: " + id);
        }
    }
}
