package ru.yandex.personal.mailimport.service;

import java.io.UnsupportedEncodingException;
import java.sql.Date;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import javax.mail.Address;
import javax.mail.BodyPart;
import javax.mail.Flags;
import javax.mail.Folder;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;

import com.sun.mail.imap.IMAPFolder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.personal.mailimport.model.MailFolder;
import ru.yandex.personal.mailimport.model.MailMessage;
import ru.yandex.personal.mailimport.model.MailMessageBodyPiece;
import ru.yandex.personal.mailimport.model.MailPerson;

public class ImapMailbox {
    private static final Logger LOG = LoggerFactory.getLogger(ImapMailbox.class);

    private final ImapConnection connection;


    public ImapMailbox(ImapConnection connection) {
        this.connection = connection;
    }

    public List<String> listFolders() {
        try {
            Folder root = connection.getStore().getDefaultFolder();
            List<String> folders = new ArrayList<>();

            LinkedList<Folder> queue = new LinkedList<>();
            queue.add(root);

            while (!queue.isEmpty()) {
                Folder f = queue.removeFirst();
                folders.add(f.getFullName());

                for (Folder child : f.list()) {
                    queue.addFirst(child);
                }
            }
            return folders;
        } catch (MessagingException e) {
            throw new RuntimeException(e);
        }
    }

    public void clear() {
        try {
            LOG.trace("Clearing mailbox");
            Folder root = connection.getStore().getDefaultFolder();
            LinkedList<Folder> deleteStack = new LinkedList<>(Arrays.asList(root.list()));
            LOG.trace("Root folder list: " + deleteStack);
            while (!deleteStack.isEmpty()) {
                Folder f = deleteStack.getFirst();
                LOG.trace("Deleting " + f.getFullName());
                deleteAllMessagesInFolder(f);
                if (f.delete(true) || f.list().length == 0) {
                    LOG.trace("folder is empty and deleted or undeletable");
                    deleteStack.removeFirst();
                } else {
                    LOG.trace("folder has child folders");
                    for (Folder f2 : f.list()) {
                        LOG.trace("Adding folder to delete " + f.getFullName());
                        deleteStack.addFirst(f2);
                    }
                }
            }
        } catch (MessagingException e) {
            e.printStackTrace();
        }
    }

    private void deleteAllMessagesInFolder(Folder f) {
        try {
            LOG.trace("Deleting all messages in folder " + f.getFullName());
            if (f.getMessageCount() != 0) {
                f.open(Folder.READ_WRITE);
                try (f) {
                    Message[] messages = f.getMessages();
                    LOG.trace("Messages in folder: " + messages.length);
                    for (Message m : messages) {
                        LOG.trace("Deleting message " + m.getSubject());
                        m.setFlag(Flags.Flag.DELETED, true);
                    }
                }
            }

        } catch (MessagingException e) {
            LOG.error("Error deleting messages in folder", e);
            e.printStackTrace();
        }
    }

    public void addFolder(MailFolder mailFolder) {
        LOG.trace("Adding folder to mailbox " + mailFolder);
        IMAPFolder folder = (IMAPFolder) getFolder(mailFolder.getName());
        Message[] messages = mailFolder
                .getMessages()
                .stream()
                .map(this::mailMessageToMessage)
                .toArray(Message[]::new);

        LOG.trace("Mapped " + messages.length + " messages");

        try (folder) {
            LOG.trace("Sending messages via IMAP");
            folder.open(Folder.READ_WRITE);
            folder.addMessages(messages);
        } catch (MessagingException e) {
            LOG.error("Error sending messages", e);
            throw new RuntimeException(e);
        }
    }

    private Folder getFolder(String name) {
        try {
            LOG.trace("Getting folder: " + name);
            Folder root = connection.getStore().getDefaultFolder();
            root.close();
            Folder newFolder = root.getFolder(name);
            if (!newFolder.exists()) {
                LOG.trace("Folder does not exist, creating " + name);
                newFolder.create(Folder.HOLDS_MESSAGES);
            }
            return newFolder;
        } catch (MessagingException e) {
            LOG.error("Can not create folder with name " + name, e);
            throw new RuntimeException(e);
        }
    }

    private Message mailMessageToMessage(MailMessage mailMessage) {
        Message result = new MimeMessage(connection.getSession());
        try {
            LOG.trace("Converting message to MIME message");

            result.setSubject(mailMessage.getSubject());
            Address from = new InternetAddress(mailMessage.getSender().getEmail(), mailMessage.getSender().getName());
            result.setFrom(from);

            Address[] tos = personsToAddressArray(mailMessage.getToReceivers());
            result.setRecipients(Message.RecipientType.TO, tos);

            Address[] ccs = personsToAddressArray(mailMessage.getCcReceivers());
            result.setRecipients(Message.RecipientType.CC, ccs);

            Address[] bccs = personsToAddressArray(mailMessage.getBccReceivers());
            result.setRecipients(Message.RecipientType.BCC, bccs);

            result.setSentDate(Date.from(mailMessage.getTimestamp()));

            if (mailMessage.getTextBody() != null) {
                result.setText(mailMessage.getTextBody());
            }

            if (mailMessage.getBody() != null) {
                Multipart content = new MimeMultipart();
                for (MailMessageBodyPiece piece : mailMessage.getBody().getPieces()) {
                    BodyPart bodyPart = new MimeBodyPart();
                    bodyPart.setText(piece.getText());

                    content.addBodyPart(bodyPart);
                }
                result.setContent(content);
            }


        } catch (MessagingException | UnsupportedEncodingException e) {
            LOG.error("Error converting message", e);
            throw new RuntimeException(e);
        }
        return result;
    }

    private Address[] personsToAddressArray(List<MailPerson> personList) {
        if (personList == null) {
            return new Address[0];
        }
        return personList.stream()
                .map(this::personToInternetAddress)
                .toArray(Address[]::new);
    }

    private InternetAddress personToInternetAddress(MailPerson mailPerson) {
        try {
            return new InternetAddress(mailPerson.getEmail(), mailPerson.getName());
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
    }
}
