#pragma once

/**
 * Boost.Asio-like "handler - async_result" concept implementation.
 * This implements coroutines.
 */
#include <io_result/async_result.h>
#include <io_result/hooks.h>
#include <io_result/error_code.h>
#include <boost/asio/spawn.hpp>
#include <boost/iterator/iterator_facade.hpp>
#include <boost/range.hpp>

namespace io_result {

/**
 * Boost.Asio basic_yield_context analog. Stores spawned with boost::asio::spawn
 * coroutine context.
 */
template <typename Handler>
class basic_yield_context {
public:
    using base_type = boost::asio::basic_yield_context<Handler>;
    using callee_type = typename base_type::callee_type;
    using caller_type = typename base_type::caller_type;

    explicit basic_yield_context(base_type base) : base_(std::move(base)) {}

    /**
     * Captures the error_code variable which is being set if an error occurs
     * Initially it is not set and the system_error will be thrown in case of
     * error.
     */
    basic_yield_context operator[](error_code& ec) const noexcept {
        basic_yield_context tmp(*this);
        tmp.ec_ = &ec;
        return tmp;
    }

    base_type base_;
    error_code* ec_ = nullptr;
};


template <typename T>
struct is_yield_context : std::false_type {};
template <typename T>
struct is_yield_context<io_result::basic_yield_context<T>> : std::true_type {};
template <typename T>
using is_yield_context_t = typename is_yield_context<std::decay_t<T>>::type;

/**
 *  Implements iterator adaptor for coroutine sequence results.
 *  Is commonly used as an iterator of range which represents sequence requests
 *  result.
 */
template <typename Iterable>
class coro_iterator : public boost::iterator_facade<
                        coro_iterator<Iterable>,
                        typename std::remove_reference<
                            decltype(std::declval<Iterable>().value())>::type,
                        boost::single_pass_traversal_tag> {
    std::shared_ptr<Iterable> src;

    friend class boost::iterator_core_access;

    void increment() {
        if(src && !src->next()) {
            src.reset();
        }
    }

    bool equal(coro_iterator const& other) const {
        return this->src == other.src;
    }

    auto& dereference() const { return src->value(); }
public:
    coro_iterator() = default;
    explicit coro_iterator(std::shared_ptr<Iterable> src) : src(std::move(src)) {
        increment();
    }
};

namespace detail {

template <typename Handler>
using callee_type = typename basic_yield_context<Handler>::callee_type;

template <typename Handler>
using caller_type = typename basic_yield_context<Handler>::caller_type;

/**
 * States for the call flow. It helps to determine when the handler has been
 * called from the method caller's coroutine. Originally there is no such
 * problem in ASIO. But in this case we can not guarantee what handler to be
 * invoked through the io_service::post strictly. It is because macs itself
 * do not operates with ASIO primitives.
 *
 * Sequence "init->result_get_called" indicates the normal flow with context
 * switch between caller coroutine and ASIO io_service event loop.
 *
 * Sequence "init->handler_called" indicates what the handler has been called
 * from the caller's coroutine without posting any task into the ASIO io_service
 * event loop.
 *
 * These two cases must be handled with different logic. In the second case
 * for sequence we need to post handlers with arguments into event loop. For
 * single values can be passed without switching between coroutines.
 */
enum class call_state {
    init, //< no calls was made
    handler_called, //< handler operator() was called
    result_get_called, //< result::get() was called
    complete, //< call sequence is complete - no more data
};

///Indicates if handler already has been called synchronously
constexpr inline bool handler_has_been_called_synchronously(call_state state) {
    return state == detail::call_state::handler_called;
}

///Indicates if handler is calling asynchronously
constexpr inline bool handler_is_calling_asynchronously(call_state state) {
    return state == call_state::result_get_called;
}

///Indicates if handler will not be called anymore for sequence
constexpr inline bool sequence_complete(call_state state) {
    return state == call_state::complete;
}

/**
 * Dispatches control flow to callee (coroutine) in case of asynchronous call.
 * If call is not asynchronous it sets up the state into handler_called to
 * indicate synchronous handler call.
 */
template <typename Handler>
void dispatch(Handler&, call_state& state, callee_type<Handler>& coro) {
    if(handler_is_calling_asynchronously(state)) {
        coro();
    } else {
        state = call_state::handler_called;
    }
}

/**
 * Dispatches control flow to caller (io_service::run()) in case of asynchronous
 * call and sets up state into the result_get_called to indicate asynchronous
 * calling flow for the handler.
 */
template <typename Handler>
void dispatch(call_state& state, caller_type<Handler>& ca) {
    if(!handler_has_been_called_synchronously(state)) {
        state = detail::call_state::result_get_called;
        ca();
    }
}

///Handler implementation for methods with return value for a single result of request
template<typename Handler, typename T>
class coro_handler {
public:
    explicit coro_handler(basic_yield_context<Handler> ctx)
    : coro_(ctx.base_.coro_.lock()), ca_(ctx.base_.ca_), handler_(ctx.base_.handler_), ec_(ctx.ec_) {}

    void operator()(error_code ec, T value) {
        *ec_ = std::move(ec);
        *value_ = std::move(value);
        dispatch(handler_, *state_, *coro_);
    }

    std::shared_ptr<callee_type<Handler>> coro_;
    caller_type<Handler>& ca_;
    Handler handler_;
    error_code* ec_ = nullptr;
    T* value_ = nullptr;
    call_state* state_ = nullptr;
};

///Handler implementation for methods with return value for a sequence result of request.
template<typename Handler, typename T>
class coro_sequence_handler {
    using Cursor = hooks::detail::Cursor<T>;
public:
    explicit coro_sequence_handler(basic_yield_context<Handler> ctx)
    : coro_(ctx.base_.coro_.lock()), ca_(ctx.base_.ca_), handler_(ctx.base_.handler_), ec_(ctx.ec_) {}

    void operator()(error_code ec, Cursor c) {
        if(!handler_is_calling_asynchronously(*state_)) {
            boost::asio::post(io_result::bind(*this, std::move(ec), std::move(c)));
        } else {
            *ec_ = std::move(ec);
            *cursor_ = std::move(c);
            dispatch(handler_, *state_, *coro_);
        }
    }

    std::shared_ptr<callee_type<Handler>> coro_;
    caller_type<Handler>& ca_;
    Handler handler_;
    error_code* ec_ = nullptr;
    Cursor* cursor_ = nullptr;
    call_state* state_ = nullptr;
};

///Handler implementation for methods with return no value
template<typename Handler>
class coro_handler<Handler, void> {
public:
    explicit coro_handler(basic_yield_context<Handler> ctx)
     : coro_(ctx.base_.coro_.lock()), ca_(ctx.base_.ca_), handler_(ctx.base_.handler_), ec_(ctx.ec_) {}

    void operator()(error_code ec) {
        *ec_ = std::move(ec);
        dispatch(handler_, *state_, *coro_);
    }

    std::shared_ptr<callee_type<Handler>> coro_;
    caller_type<Handler>& ca_;
    Handler handler_;
    error_code* ec_ = nullptr;
    call_state* state_ = nullptr;
};

} // namespace detail

template<typename Handler>
struct handler_type<basic_yield_context<Handler>, Hook<void>> {
    using type = detail::coro_handler<Handler, void>;
};

template<typename Handler, typename T>
struct handler_type<basic_yield_context<Handler>, Hook<Sequence<T>>> {
    using type = detail::coro_sequence_handler<Handler, T>;
};

template<typename Handler, typename T>
struct handler_type<basic_yield_context<Handler>, Hook<T>> {
    using type = detail::coro_handler<Handler, typename Hook<T>::second_argument_type>;
};

template<typename Handler, typename T>
class async_result<detail::coro_handler<Handler, T>> {
public:
    explicit async_result(detail::coro_handler<Handler, T>& h)
    : handler_(h), ca_(h.ca_), out_ec_(h.ec_) {

        if(!out_ec_) {
            h.ec_ = &ec_;
        }
        h.value_ = &value_;
        h.state_ = &state_;
    }

    auto get() {
        handler_.coro_.reset(); // Must not hold shared_ptr to coro while suspended.
        detail::dispatch<Handler>(state_, ca_);
        if (!out_ec_ && ec_) {
            throw system_error(std::move(ec_));
        }
        return std::move(value_);
    }

private:
    detail::coro_handler<Handler, T>& handler_;
    detail::caller_type<Handler>& ca_;
    error_code* out_ec_;
    error_code ec_;
    T value_;
    detail::call_state state_ = detail::call_state::init;
};

template<typename Handler, typename T>
class async_result<detail::coro_sequence_handler<Handler, T>> {
public:
    struct result {
        explicit result(detail::coro_sequence_handler<Handler, T>& h)
        : handler_(h.handler_), ca_(h.ca_), out_ec_(h.ec_) {

            if(!out_ec_) {
                h.ec_ = &ec_;
            }
            h.cursor_ = &cursor_;
            h.state_ = &state_;
        }

        ~result() {
            // Since cursor calls resume() automatically at the destructor
            // we need to prevent it because at this point we do not want to
            // resume with continuation. This point means all iterators of the
            // range has been destoyed and operation must be cancelled. The
            // continuation will be destroyed, and it means what operation
            // is cancelled.
            cursor_.release();
        }

        bool next() {
            if (detail::sequence_complete(state_)) {
                throw std::out_of_range("call for next element in complete sequence");
            }

            // This post is to avoid recursive coro_sequence_handler call
            // within the continuation.
            if(cursor_.resumable()) {
                cursor_.resume(boost::asio::get_associated_executor(handler_));
            }

            detail::dispatch<Handler>(state_, ca_);

            if (ec_ || !cursor_) {
                state_ = detail::call_state::complete;
            }

            if (!out_ec_ && ec_) {
                throw system_error(std::move(ec_));
            }

            return !detail::sequence_complete(state_);
        }

        auto& value() { return cursor_.get(); }

    private:
        Handler handler_;
        detail::caller_type<Handler>& ca_;
        error_code* out_ec_;
        error_code ec_;
        hooks::detail::Cursor<T> cursor_;
        detail::call_state state_ = detail::call_state::init;
    };

    explicit async_result(detail::coro_sequence_handler<Handler, T>& h)
    : res(std::make_shared<result>(h)), handler_(h) {}

    auto get() {
        using Iter = coro_iterator<result>;
        handler_.coro_.reset(); // Must not hold shared_ptr to coro while suspended.
        return boost::make_iterator_range(Iter{res}, Iter{});
    }

private:
    std::shared_ptr<result> res;
    detail::coro_sequence_handler<Handler, T>& handler_;
};

template<typename Handler>
class async_result<detail::coro_handler<Handler, void> > {
public:
    explicit async_result(detail::coro_handler<Handler, void>& h)
    : handler_(h), ca_(h.ca_), out_ec_(h.ec_) {

        if(!out_ec_) {
            h.ec_ = &ec_;
        }
        h.state_ = &state_;
    }

    void get() {
        handler_.coro_.reset(); // Must not hold shared_ptr to coro while suspended.
        detail::dispatch<Handler>(state_, ca_);
        if (!out_ec_ && ec_) {
            throw system_error(std::move(ec_));
        }
    }

private:
    detail::coro_handler<Handler, void>& handler_;
    detail::caller_type<Handler>& ca_;
    error_code* out_ec_;
    error_code ec_;
    detail::call_state state_ = detail::call_state::init;
};


template <typename Handler>
auto make_yield_context(boost::asio::basic_yield_context<Handler> yield) {
    return basic_yield_context<Handler>(yield);
}

template <typename Handler>
auto make_yield_context(boost::asio::basic_yield_context<Handler> yield, error_code& ec) {
    return make_yield_context(yield)[ec];
}

template <typename Handler>
auto make_yield_context(basic_yield_context<Handler> yield) {
    return yield;
}

template <typename Handler>
auto make_yield_context(basic_yield_context<Handler> yield, error_code& ec) {
    return yield[ec];
}

} // namespace io_result

namespace boost::asio {

template <typename Handler, typename T, typename Allocator>
struct associated_allocator<::io_result::detail::coro_handler<Handler, T>, Allocator>
{
    typedef typename associated_allocator<Handler, Allocator>::type type;

    static type get(const ::io_result::detail::coro_handler<Handler, T>& h,
        const Allocator& a = Allocator()) BOOST_ASIO_NOEXCEPT
    {
        return associated_allocator<Handler, Allocator>::get(h.handler_, a);
    }
};

template <typename Handler, typename T, typename Executor>
struct associated_executor<::io_result::detail::coro_handler<Handler, T>, Executor>
{
    typedef typename associated_executor<Handler, Executor>::type type;

    static type get(const ::io_result::detail::coro_handler<Handler, T>& h,
        const Executor& ex = Executor()) BOOST_ASIO_NOEXCEPT
    {
        return associated_executor<Handler, Executor>::get(h.handler_, ex);
    }
};

template <typename Handler, typename T, typename Allocator>
struct associated_allocator<::io_result::detail::coro_sequence_handler<Handler, T>, Allocator>
{
    typedef typename associated_allocator<Handler, Allocator>::type type;

    static type get(const ::io_result::detail::coro_sequence_handler<Handler, T>& h,
        const Allocator& a = Allocator()) BOOST_ASIO_NOEXCEPT
    {
        return associated_allocator<Handler, Allocator>::get(h.handler_, a);
    }
};

template <typename Handler, typename T, typename Executor>
struct associated_executor<::io_result::detail::coro_sequence_handler<Handler, T>, Executor>
{
    typedef typename associated_executor<Handler, Executor>::type type;

    static type get(const ::io_result::detail::coro_sequence_handler<Handler, T>& h,
        const Executor& ex = Executor()) BOOST_ASIO_NOEXCEPT
    {
        return associated_executor<Handler, Executor>::get(h.handler_, ex);
    }
};

} // namespace boost::asio
