#pragma once

#include <io_result/error_code.h>
#include <io_result/bind.h>
#include <boost/function.hpp>
#include <boost/optional.hpp>
#include <boost/asio/post.hpp>
#include <boost/asio/dispatch.hpp>
#include <list>
#include <optional>
#include <tuple>

namespace io_result {
namespace hooks {

template <typename... Args>
using function = ::std::function<Args...>;

template <typename T>
struct Sequence;

template <typename T>
class Optional;

template <typename T>
struct is_optional : std::false_type {};

template <typename T>
struct is_optional<Optional<T>> : std::true_type {};

template <typename T>
class Optional final {
public:
    using value_type = T;

    Optional() = default;
    Optional(const Optional&) = default;
    Optional(Optional&&) = default;
    Optional(boost::optional<T> value) : value(std::move(value)) {}
    Optional(T value) : value(std::move(value)) {}

    Optional& operator= (const Optional&) = default;
    Optional& operator= (Optional&&) = default;

    template <typename U>
    std::enable_if_t<!is_optional<std::decay_t<U>>::value, Optional&>
    operator= (U&& value) {
        this->value = std::forward<U>(value);
        return *this;
    }

    operator bool() const noexcept {
        return value.is_initialized();
    }

    operator boost::optional<T>&() & noexcept {
        return value;
    }

    operator const boost::optional<T>&() const & noexcept {
        return value;
    }

    operator boost::optional<T>&&() && noexcept {
        return std::move(value);
    }

    decltype(auto) operator*() noexcept {
        return *value;
    }

    decltype(auto) operator*() const noexcept {
        return *value;
    }

    auto* operator->() noexcept {
        return value.operator->();
    }

    auto* operator->() const noexcept {
        return value.operator->();
    }

    bool isInitialized() const noexcept {
        return value.is_initialized();
    }

    template <typename U>
    decltype(auto) getValueOr(U&& v) noexcept {
        return value.get_value_or(std::forward<U>(v));
    }

    template <typename U>
    decltype(auto) getValueOr(U&& v) const noexcept {
        return value.get_value_or(std::forward<U>(v));
    }

    template <typename U>
    bool operator== (const Optional<U>& that) const noexcept {
        return value == that.value;
    }

    template <typename U>
    bool operator!= (const Optional<U>& that) const noexcept {
        return value != that.value;
    }

    template <typename U>
    std::enable_if_t<!is_optional<U>::value, bool>
    operator== (const U& that) const noexcept {
        return value == that;
    }

    template <typename U>
    std::enable_if_t<!is_optional<U>::value, bool>
    operator!= (const U& that) const noexcept {
        return value != that;
    }

    auto flatten() const & {
        return flatten(value);
    }

    auto flatten() && {
        return flatten(std::move(value));
    }

private:
    template <typename O>
    static auto flatten(O&& opt) {
        using Type = std::decay_t<decltype(**opt)>;
        if (opt && *opt) {
            return Optional<Type>(std::move(**opt));
        } else {
            return Optional<Type>();
        }
    }

    boost::optional<T> value;
};

template <typename T>
Optional<std::decay_t<T>> makeOptional(T&& value) {
    return Optional<std::decay_t<T>>(std::forward<T>(value));
}

template <typename T, typename U>
std::enable_if_t<!is_optional<T>::value, bool>
operator== (const T& value, const Optional<U>& opt) noexcept {
    return opt == value;
}

template <typename T, typename U>
std::enable_if_t<!is_optional<T>::value, bool>
operator!= (const T& value, const Optional<U>& opt) noexcept {
    return opt != value;
}

namespace detail {

template <typename T>
struct StrategySelector;

template <typename Arg = void>
struct HookImpl;

}

struct Strategy {
    // Handler will be called once with value or error
    struct Once;
    // Handler will be called 1..N+1 times with empty value at last or with error
    struct Sequence;
    // Handler will be called once with value or empty value or error
    struct Optional;

    template <typename T>
    using type = typename detail::StrategySelector<T>::type;
};

namespace detail {

template <typename T>
struct StrategySelector<HookImpl<T>> {
    using type = Strategy::Once;
};
template <typename T>
struct StrategySelector<HookImpl<Sequence<T>>> {
    using type = Strategy::Sequence;
};
template <typename T>
struct StrategySelector<HookImpl<Optional<T>>> {
    using type = Strategy::Optional;
};

template <typename T, typename Self>
using disable_if_t = typename std::enable_if<
        !std::is_same<typename std::decay<T>::type, Self>::value &&
        !std::is_same<typename std::decay<T>::type, std::tuple<Self&&>>::value, void>::type;



template <>
struct HookImpl<void> {
    using value_type = void;
    using error_code = io_result::error_code;
    using argument_type = error_code;
    using result_type = void;

private:
    struct Func {
        virtual void invoke(error_code) = 0;
        virtual ~Func() = default;
    };

    template <typename Handler>
    struct FuncImpl : Func {
        Handler handler_;
        FuncImpl(Handler h) : handler_(std::move(h)) {}
        void invoke(error_code ec) override {
            boost::asio::dispatch(io_result::bind(std::move(handler_), std::move(ec)));
        }
    };

    std::shared_ptr<Func> h = nullptr;

public:
    HookImpl(const HookImpl& ) = default;
    HookImpl(HookImpl&& ) = default;
    HookImpl() = default;

    template <typename T>
    HookImpl(T v, disable_if_t<T, HookImpl>* = nullptr)
     : h(std::make_shared<FuncImpl<T>>(std::move(v))) {}
    void operator()(error_code e = error_code()) const { h->invoke(std::move(e)); }
    bool operator !() const noexcept { return !h;}
};

template <typename T>
struct Argument { using type = T; };

template <typename T>
struct Argument<Optional<T>> { using type = boost::optional<T>; };

template <typename T>
class Cursor {
public:
    using value_type = T;
    using Continuation = std::function<void()>;
    using Context = std::optional<std::tuple<T, Continuation>>;

    T& get() { return std::get<T>(**v_); }
    const T& get() const { return std::get<T>(**v_); }
    T* operator -> () { return &get(); }
    const T* operator -> () const { return &get(); }
    T& operator * () { return get(); }
    const T& operator * () const { return get(); }

    operator bool () const noexcept { return v_ && *v_; }

    void resume() {
        auto c = std::move(continuation());
        c();
    }

    template <typename Executor>
    void resume(const Executor& ex) {
        boost::asio::post(ex, std::move(continuation()));
    }

    Context release() {
        if (!v_) {
            return std::nullopt;
        }
        auto retval = std::move(*v_);
        *v_ = std::nullopt;
        return retval;
    }

    bool resumable() const noexcept {
        return v_ && *v_ && continuation();
    }

    Cursor() = default;
    Cursor(T v, Continuation c)
    : v_(std::make_shared<Context>(std::make_tuple(std::move(v), std::move(c)))) {
        if (!continuation()) {
            throw std::invalid_argument("io_result::Cursor() null continuation");
        }
    }
    ~Cursor() noexcept {
        if (resumable() && v_.use_count() == 1) try {
            resume();
        } catch (...) {
        }
    }

    Cursor(const Cursor&) = default;
    Cursor(Cursor&& src) : v_(std::move(src.v_)) {
        src.v_ = nullptr;
    }
    Cursor& operator = (const Cursor&) = default;
    Cursor& operator = (Cursor&& src) {
        v_ = std::move(src.v_);
        src.v_ = nullptr;
        return *this;
    }

private:
    Continuation& continuation() {
        return std::get<Continuation>(**v_);
    }
    const Continuation& continuation() const {
        return std::get<Continuation>(**v_);
    }
    std::shared_ptr<Context> v_ = nullptr;
};

template <typename T>
struct Argument<Sequence<T>> { using type = Cursor<T>; };

template <typename ArgT>
struct HookImpl {
    using value_type = ArgT;
    using Arg = typename Argument<ArgT>::type;
    using error_code = io_result::error_code;
    using second_argument_type = Arg;
    using first_argument_type = error_code;
    using result_type = void;

private:
    struct Func {
        virtual void invoke(error_code, Arg) = 0;
        virtual ~Func() = default;
    };

    template <typename Handler, typename>
    struct FuncImpl : Func {
        Handler handler_;
        FuncImpl(Handler h) : handler_(std::move(h)) {}
        void invoke(error_code ec, Arg arg) override {
            boost::asio::dispatch(io_result::bind(std::move(handler_), std::move(ec), std::move(arg)));
        }
    };

    template <typename Handler, typename T>
    struct FuncImpl<Handler, Sequence<T>> : Func {
        Handler handler_;
        FuncImpl(Handler h) : handler_(std::move(h)) {}
        void invoke(error_code ec, Arg arg) override {
            boost::asio::dispatch(io_result::bind(handler_, std::move(ec), std::move(arg)));
        }
    };

    std::shared_ptr<Func> h = nullptr;

public:
    HookImpl(const HookImpl& ) = default;
    HookImpl(HookImpl&& ) = default;
    HookImpl() = default;

    template <typename T>
    HookImpl(T v, disable_if_t<T, HookImpl>* = nullptr)
    : h(std::make_shared<FuncImpl<T, ArgT>>(std::move(v))) {}

    void operator()(error_code e = error_code(), Arg arg = Arg()) const {
        h->invoke(std::move(e), std::forward<Arg>(arg));
    }

    void operator()(Arg arg) const {
        (*this)(error_code(), std::forward<Arg>(arg));
    }

    bool operator !() const noexcept { return !h;}
};

} // namespace detail

namespace export_ {

template <typename T = void>
using Hook = detail::HookImpl<T>;

}// namespace export_

using namespace export_;

}

using namespace hooks::export_;
using hooks::Optional;
using hooks::Sequence;
using hooks::detail::Cursor;

using hooks::makeOptional;

}
