#pragma once

/**
 * This header is forced copy-paste adoptation of the boost/asio/use_future.hpp
 * The reason is custom error_code type.
 */

#include <io_result/async_result.h>
#include <io_result/hooks.h>
#include <boost/asio/use_future.hpp>
#include <future>

namespace io_result {

using ::boost::asio::use_future_t;

using ::boost::asio::use_future;

namespace detail {

// Completion handler to adapt a promise as a completion handler.
template<typename T>
class promise_handler {
public:
    // Construct from use_future special value.
    template<typename Alloc>
    promise_handler(use_future_t<Alloc> uf)
    : promise_ (std::allocate_shared<std::promise<T> >(
            typename Alloc::template rebind<char>::other(
                    uf.get_allocator()), std::allocator_arg,
            typename Alloc::template rebind<char>::other(
                    uf.get_allocator()))) {
    }

    void operator()(error_code ec, T t) {
        if (ec) {
            promise_->set_exception(std::make_exception_ptr(system_error(ec)));
        } else {
            promise_->set_value(std::move(t));
        }
    }

    std::shared_ptr<std::promise<T>> promise_;
};

// Completion handler to adapt a void promise as a completion handler.
template<>
class promise_handler<void> {
public:
    // Construct from use_future special value. Used during rebinding.
    template<typename Alloc>
    promise_handler(use_future_t<Alloc> uf)
    : promise_ (std::allocate_shared<std::promise<void> >(
            typename Alloc::template rebind<char>::other(
                    uf.get_allocator()), std::allocator_arg,
            typename Alloc::template rebind<char>::other(
                    uf.get_allocator()))) {
    }

    void operator()(error_code ec) {
        if (ec) {
            promise_->set_exception(std::make_exception_ptr(system_error(ec)));
        } else {
            promise_->set_value();
        }
    }

    std::shared_ptr<std::promise<void>> promise_;
};

} // namespace detail


template<typename Allocator, typename T>
struct handler_type<use_future_t<Allocator>, Hook<T>> {
    using type = detail::promise_handler<typename Hook<T>::second_argument_type>;
};

template<typename Allocator>
struct handler_type<use_future_t<Allocator>, Hook<void>> {
    using type = detail::promise_handler<void>;
};

// Handler traits specialisation for promise_handler.
template<typename T>
class async_result<detail::promise_handler<T> > {
public:
    // Constructor creates a new promise for the async operation, and obtains the
    // corresponding future.
    explicit async_result(detail::promise_handler<T>& h)
    : value_(h.promise_->get_future()) {
    }

    // Obtain the future to be returned from the initiating function.
    auto get() { return std::move(value_); }

private:
    std::future<T> value_;
};

} // namespace io_result
