#pragma once

/**
 * Boost.Asio-like "handler - async_result" concept implementation.
 * This implements synchronization via std::promise/std::future.
 */
#include <io_result/async_result.h>
#include <io_result/hooks.h>
#include <boost/mpl/has_xxx.hpp>
#include <future>
#include <iterator>

namespace io_result {

template <typename Handler>
class basic_sync_context{
public:
    /**
     * Captures the error_code variable which is being set if an error occurs
     * Initially it is not set and the system_error will be thrown in case of
     * error.
     */
    basic_sync_context operator[](error_code& ec) const noexcept {
        basic_sync_context tmp(*this);
        tmp.ec_ = &ec;
        return tmp;
    }

    error_code* ec_ = nullptr;
};

using sync_context = basic_sync_context<void>;

constexpr sync_context use_sync;

namespace detail {

/// Safely sets system_error exception with given error_code for the promise p
template <typename T>
inline void set_exception(std::promise<T>&& p, error_code ec) {
    std::promise<T> local{std::move(p)};
    local.set_exception(std::make_exception_ptr(system_error(std::move(ec))));
}

/// Safely sets value v for the promise p
template <typename T, typename ... Value>
inline void set_value(std::promise<T>&& p, Value&&... v) {
    std::promise<T> local{std::move(p)};
    local.set_value(std::forward<Value>(v)...);
}

/// Performs error forwarding algorithm
/**
 * This function implements error forwarding rule of the basic_sync_context<>
 * for the handlers which use std::promise<void> for synchronization.
 * If ec_out is set and not null the error_code ec will be placed into *ec_out
 * and promise p's value will be set. In other case - it will be placed into
 * the promise p like system_error exception.
 */
inline void forward_error(std::promise<void>&& p, error_code* ec_out, error_code ec) {
    if(!ec_out) {
        set_exception(std::move(p), std::move(ec));
    } else {
        *ec_out = std::move(ec);
        set_value(std::move(p));
    }
}

/// Handler for the single value requests
template<typename Handler, typename T>
class sync_handler {
public:
    explicit sync_handler(const Handler& h) : ec_out_(h.ec_) {}

    void operator()(error_code ec, T value) {
        if(ec) {
            forward_error(std::move(*p_), ec_out_, std::move(ec));
        } else {
            *value_ = std::move(value);
            set_value(std::move(*p_));
        }
    }

    std::promise<void>* p_ = nullptr;
    T* value_ = nullptr;
    error_code* ec_out_ = nullptr;
};

/// Handler for the sequence value requests and inserter (iterator) as Handler
template<typename Inserter, typename T>
class sync_sequence_handler {
public:
    explicit sync_sequence_handler(Inserter out) : out_(out) {}

    void operator()(error_code ec, hooks::detail::Cursor<T> value) {
        if(ec) {
            set_exception(std::move(*p_), std::move(ec));
        } else if(value) {
            *out_++ = std::move(*value);
        } else {
            set_value(std::move(*p_), out_);
        }
    }

    Inserter out_;
    std::promise<Inserter>* p_ = nullptr;
};

/// Handler for the sequence value requests and result is returned as container
template<typename T>
class sync_sequence_handler<sync_context, T> {
public:
    explicit sync_sequence_handler(const sync_context& h) : ec_out_(h.ec_) {}

    void operator()(error_code ec, hooks::detail::Cursor<T> value) {
        if(ec) {
            forward_error(std::move(*p_), ec_out_, std::move(ec));
        } else if(value) {
            value_->emplace_back(std::move(*value));
        } else {
            set_value(std::move(*p_));
        }
    }
    std::list<T>* value_ = nullptr;
    std::promise<void>* p_ = nullptr;
    error_code* ec_out_ = nullptr;
};

/// Handler for requests without return value
template<typename Handler>
class sync_handler<Handler, void> {
public:
    explicit sync_handler(const Handler& h) : ec_out_(h.ec_) {}

    void operator()(error_code ec) {
        if(ec) {
            forward_error(std::move(*p_), ec_out_, std::move(ec));
        } else {
            set_value(std::move(*p_));
        }
    }

    std::promise<void>* p_ = nullptr;
    error_code* ec_out_ = nullptr;
};

} // namespace detail

template<typename T>
struct handler_type<sync_context, Hook<T>> {
    using type = detail::sync_handler<sync_context, typename Hook<T>::second_argument_type>;
};

template<>
struct handler_type<sync_context, Hook<void>> {
    using type = detail::sync_handler<sync_context, void>;
};

template<typename T>
struct handler_type<sync_context, Hook<Sequence<T>>> {
    using type = detail::sync_sequence_handler<sync_context, T>;
};

template<typename Container, typename T>
struct handler_type<std::back_insert_iterator<Container>, Hook<Sequence<T>>> {
    using type = detail::sync_sequence_handler<std::back_insert_iterator<Container>, T>;
};

template<typename Container, typename T>
struct handler_type<std::insert_iterator<Container>, Hook<Sequence<T>>> {
    using type = detail::sync_sequence_handler<std::insert_iterator<Container>, T>;
};

BOOST_MPL_HAS_XXX_TRAIT_DEF(iterator_category)

template<typename Handler>
struct handler_type<Handler, Hook<Sequence<
                                typename std::enable_if<
                                    has_iterator_category<Handler>::value,
                                    typename Handler::value_type>::type>>> {
    using type = detail::sync_sequence_handler<Handler, typename Handler::value_type>;
};

template<typename Handler, typename T>
class async_result<detail::sync_handler<Handler, T>> {
public:
    explicit async_result(detail::sync_handler<Handler, T>& h) {
        h.p_ = &promise_;
        h.value_ = &value_;
    }

    auto get() {
        future_.get();
        return std::move(value_);
    }

private:
    T value_;
    std::promise<void> promise_;
    std::future<void> future_ = promise_.get_future();
};

template<typename Handler>
class async_result<detail::sync_handler<Handler, void>> {
public:
    explicit async_result(detail::sync_handler<Handler, void>& h) {
        h.p_ = &promise_;
    }

    void get() {
        future_.get();
    }

private:
    std::promise<void> promise_;
    std::future<void> future_ = promise_.get_future();
};

template<typename Handler, typename T>
class async_result<detail::sync_sequence_handler<Handler, T>> {
public:
    explicit async_result(detail::sync_sequence_handler<Handler, T>& h) {
        h.p_ = &promise_;
    }

    auto get() {
        return future_.get();
    }

private:
    std::promise<Handler> promise_;
    std::future<Handler> future_ = promise_.get_future();
};

template<typename T>
class async_result<detail::sync_sequence_handler<sync_context, T>> {
public:
    explicit async_result(detail::sync_sequence_handler<sync_context, T>& h) {
        h.p_ = &promise_;
        h.value_ = &value_;
    }

    auto get() {
        future_.get();
        return std::move(value_);
    }

private:
    std::list<T> value_;
    std::promise<void> promise_;
    std::future<void> future_ = promise_.get_future();
};

} // namespace io_result
