from typing import Any, Optional

from aiohttp import web

from sendr_aiohttp import BaseUrlDispatcher, create_apispec, setup_swagger_route
from sendr_aiopg.engine.multiple import MultipleEngine
from sendr_qlog.http.aiohttp import get_middleware_logging_adapter, signal_request_id_header

from mail.ipa.ipa import __version__
from mail.ipa.ipa.api.middlewares import middleware_response_formatter, middleware_stats, middleware_tvm
from mail.ipa.ipa.api.routes.routes import APP_ROUTES
from mail.ipa.ipa.api.routes.utility import UTILITY_ROUTES
from mail.ipa.ipa.conf import settings
from mail.ipa.ipa.interactions.base import BaseInteractionClient, create_connector
from mail.ipa.ipa.utils.db import create_configured_engine


class IpaApplication(web.Application):
    _urls = (
        APP_ROUTES,
        UTILITY_ROUTES,
    )

    def __init__(self, db_engine: Optional[MultipleEngine] = None):
        super().__init__(
            router=BaseUrlDispatcher(),
            middlewares=(
                middleware_stats,
                get_middleware_logging_adapter(),
                middleware_response_formatter,
                middleware_tvm,
            ),
            client_max_size=settings.AIOHTTP_MAX_BODY_SIZE,
        )

        if db_engine:
            self.db_engine = db_engine
        else:
            self.on_startup.append(self.open_engine)
            self.on_cleanup.append(self.close_engine)

        self.add_routes()
        self.add_sentry()
        self.on_response_prepare.append(signal_request_id_header)
        self.on_startup.append(self.create_connector)
        self.on_cleanup.append(self.close_engine)
        self.on_cleanup.append(self.close_connector)

    def add_routes(self) -> None:  # type: ignore
        assert isinstance(self.router, BaseUrlDispatcher)
        for routes in self._urls:
            self.router.add_routes(routes)
        if settings.ENABLE_APISPEC:
            spec = create_apispec(
                app=self,
                title='ipa',
                version=__version__,
            )
            setup_swagger_route(
                app=self,
                spec=spec,
                swagger_route='/api/swagger.json',
                swagger_route_name='swagger',
            )

    def add_sentry(self) -> None:
        from sendr_qlog.sentry import sentry_init
        if settings.SENTRY_DSN:
            sentry_init(settings.SENTRY_DSN, release=__version__)

    async def open_engine(self, _: Any) -> None:
        self.db_engine = create_configured_engine()
        self.db_engine.connect_async()

    async def close_engine(self, app: web.Application) -> None:
        if self.db_engine:
            self.db_engine.close()
        await self.db_engine.wait_closed()

    async def create_connector(self, _: Any) -> None:
        BaseInteractionClient.CONNECTOR = create_connector()

    async def close_connector(self, _: Any) -> None:
        await BaseInteractionClient.close_connector()
