from dataclasses import asdict
from typing import Any, Awaitable, Callable, Dict

from aiohttp import web

from sendr_aiohttp.handler import MethodSchema, request_schema, response_schema

from mail.ipa.ipa.api.exceptions import UnsupportedMediaTypeError
from mail.ipa.ipa.api.handlers.base import BaseHandler
from mail.ipa.ipa.api.schemas.import_ import (
    create_import_from_csv_request_schema, create_import_json_body_schema, create_import_request_schema,
    create_task_response_schema, info_import_params_schema, info_import_response_schema, stat_import_response_schema
)
from mail.ipa.ipa.api.schemas.path import org_id_schema
from mail.ipa.ipa.conf import settings
from mail.ipa.ipa.core.actions.import_.csv import CreateImportFromCSVAction
from mail.ipa.ipa.core.actions.import_.json import CreateImportFromJSONAction
from mail.ipa.ipa.core.actions.import_.stop import CreateStopImportAction
from mail.ipa.ipa.core.actions.report import WriteCSVReportAction
from mail.ipa.ipa.core.actions.stats.info import GetImportInfoAction
from mail.ipa.ipa.core.actions.stats.summary import GetImportStatAction
from mail.ipa.ipa.core.entities.import_params import GeneralInitImportParams
from mail.ipa.ipa.core.entities.password import Password
from mail.ipa.ipa.core.entities.user_info import UserInfo


class ImportHandler(BaseHandler):
    def _get_or_create_method_schema(self) -> MethodSchema:
        request_method = self.request.method.lower()
        if request_method == 'post':
            method = self._get_method_for_content_type()
            return MethodSchema.get_or_create_method_schema(method)
        return super()._get_or_create_method_schema()

    def _get_method_for_content_type(self) -> Callable[[], Awaitable[web.StreamResponse]]:
        content_type = self.request.content_type

        if content_type == 'text/csv':
            return self.from_csv
        elif content_type == 'application/json':
            return self.from_json

        raise UnsupportedMediaTypeError

    def fill_logger(self, data: Dict[str, Any]) -> None:
        self.logger.context_push(
            org_id=data['org_id'],
            admin_uid=data['admin_uid'],
            server=data['server'],
            port=data['port'],
            imap=data['imap'],
            mark_archive_read=data['mark_archive_read'],
            delete_msgs=data['delete_msgs'],
        )

    @request_schema(create_import_from_csv_request_schema, location='query')
    @request_schema(org_id_schema, location='match_info')
    @response_schema(create_task_response_schema)
    async def from_csv(self):
        data = await self.get_data()

        csv_name = data.pop('name')

        self.fill_logger(data)
        self.logger.context_push(name=csv_name)

        content = self.request.content
        task_id = await self.run_action(CreateImportFromCSVAction,
                                        import_params=GeneralInitImportParams(**data),
                                        stream=content,
                                        name=csv_name)
        return self.make_response({'data': {'task_id': task_id}})

    @request_schema(create_import_request_schema, location='query')
    @request_schema(org_id_schema, location='match_info')
    @response_schema(create_task_response_schema)
    async def from_json(self):
        data = await self.get_data()

        self.fill_logger(data)

        json_data = await self.parse_request(create_import_json_body_schema, 'json')
        task_id = await self.run_action(
            CreateImportFromJSONAction,
            params=GeneralInitImportParams(**data),
            users=[
                UserInfo(
                    login=user_json['login'],
                    password=Password.from_plain(user_json['password']),
                    src_login=user_json['src_login'],
                )
                for user_json in json_data['users']
            ],
        )
        return self.make_response({'data': {'task_id': task_id}})

    @request_schema(org_id_schema, location='match_info')
    @request_schema(info_import_params_schema, location='query')
    @response_schema(info_import_response_schema)
    async def get(self):
        data = await self.get_data()
        info, has_more = await self.run_action(
            GetImportInfoAction,
            org_id=data['org_id'],
            only_errors=data['only_errors'],
        )
        return self.make_response({
            'data': {
                'collectors': [
                    {
                        **(asdict(collector) if collector is not None else {}),
                        **asdict(user),
                        'error': error,
                    }
                    for user, collector, error in info
                ],
                'has_more': has_more,
            }
        })

    async def post(self):
        method = self._get_method_for_content_type()
        return await method()


class ReportImportHandler(BaseHandler):
    @request_schema(org_id_schema, location='match_info')
    async def get(self):
        data = await self.get_data()
        response = web.StreamResponse(headers={
            'Content-Type': 'text/csv',
            'Content-Disposition': f'attachment; filename="{settings.CSV_REPORT_FILE_NAME}"',
        })
        await response.prepare(self.request)
        await self.run_action(WriteCSVReportAction, org_id=data['org_id'], output=response)
        return response


class StatImportHandler(BaseHandler):
    @request_schema(org_id_schema, location='match_info')
    @response_schema(stat_import_response_schema)
    async def get(self):
        data = await self.get_data()
        import_stat = await self.run_action(GetImportStatAction, org_id=data['org_id'])
        return self.make_response({'data': import_stat})


class StopImportHandler(BaseHandler):
    @request_schema(org_id_schema, location='match_info')
    @response_schema(create_task_response_schema)
    async def post(self):
        data = await self.get_data()
        self.logger.context_push(org_id=data['org_id'])
        task_id = await self.run_action(CreateStopImportAction, org_id=data['org_id'])
        return self.make_response({'data': {'task_id': task_id}})
