from dataclasses import dataclass
from typing import Any, Optional, cast

from marshmallow import fields, pre_dump
from marshmallow_enum import EnumField

from mail.ipa.ipa.api.schemas.base import BasePaginatedRequestSchema, BaseSchema, BaseStatsSchema, SuccessResponseSchema
from mail.ipa.ipa.api.schemas.common import CollectorSchema
from mail.ipa.ipa.api.schemas.user import UserSchema
from mail.ipa.ipa.core.entities.collector import Collector
from mail.ipa.ipa.core.entities.enums import EventType, UserImportError
from mail.ipa.ipa.interactions.yarm.entities import YarmCollectorState


@dataclass
class _CollectorWithError(Collector):
    error: Optional[UserImportError] = None


class EventSchema(BaseSchema):
    org_id = fields.Integer()
    event_type = EnumField(EventType, by_value=True)
    revision = fields.Integer()
    data = fields.Dict()
    event_id = fields.Integer()
    created_at = fields.DateTime()


class SupportCollectorStatusesResponseSchema(SuccessResponseSchema):
    data = fields.List(fields.String())


class SupportCollectorsRequestSchema(BasePaginatedRequestSchema):
    created_from = fields.DateTime()
    created_to = fields.DateTime()
    status = fields.String()
    login = fields.String()


class YarmCollectorFolderStatus(BaseStatsSchema):
    name = fields.String()


class YarmStatusSchema(BaseStatsSchema):
    folders = fields.Nested(YarmCollectorFolderStatus, many=True)


class YarmCollectorSchema(BaseSchema):
    pop_id = fields.String()
    server = fields.String()
    port = fields.Integer()
    login = fields.String()
    ssl = fields.Boolean()
    email = fields.String()
    imap = fields.Boolean()
    state = EnumField(YarmCollectorState)
    delete_msgs = fields.Boolean()
    error_status = fields.String()

    status = fields.Nested(YarmStatusSchema)


class SupportCollectorSchema(CollectorSchema):
    collector_id = fields.Integer()
    pop_id = fields.String()

    @pre_dump
    def convert_error(self, obj: Collector, **kwargs: Any) -> Collector:
        obj_with_error: _CollectorWithError = cast(_CollectorWithError, obj)
        obj_with_error.error = UserImportError.get_error(collector_status=obj_with_error.status)
        return obj_with_error


class SupportCollectorWithYarmSchema(SupportCollectorSchema):
    yarm_collector = fields.Nested(YarmCollectorSchema)


class SupportCollectorsResponseSchema(SuccessResponseSchema):
    data = fields.Nested(SupportCollectorSchema, many=True)


class SupportCollectorResponseSchema(SuccessResponseSchema):
    data = fields.Nested(SupportCollectorWithYarmSchema)


class SupportEventsRequestSchema(BasePaginatedRequestSchema):
    event_type = EnumField(EventType, by_value=True)
    created_from = fields.DateTime()
    created_to = fields.DateTime()


class SupportEventsResponseSchema(SuccessResponseSchema):
    data = fields.Nested(EventSchema, many=True)


class SupportUsersRequestSchema(BasePaginatedRequestSchema):
    login = fields.String()


class SupportUserSchema(UserSchema):
    user_id = fields.Integer()
    suid = fields.Integer()
    created_at = fields.DateTime()
    error = fields.String()


class SupportUsersResponseSchema(SuccessResponseSchema):
    data = fields.Nested(SupportUserSchema, many=True)


class SupportCollectorUpdateRequestSchema(BaseSchema):
    enabled = fields.Boolean(required=True)


support_collectors_request_schema = SupportCollectorsRequestSchema()
support_collectors_response_schema = SupportCollectorsResponseSchema()

support_collector_update_request_schema = SupportCollectorUpdateRequestSchema()
support_collector_response_schema = SupportCollectorResponseSchema()

support_collector_statuses_response_schema = SupportCollectorStatusesResponseSchema()

support_events_request_schema = SupportEventsRequestSchema()
support_events_response_schema = SupportEventsResponseSchema()

support_users_request_schema = SupportUsersRequestSchema()
support_users_response_schema = SupportUsersResponseSchema()
