from dataclasses import asdict, is_dataclass
from typing import Any, ClassVar, Dict, Optional

from sendr_aiopg.action import BaseDBAction as BDBAction
from sendr_core.action import BaseAction as BAction
from sendr_qlog import LoggerContext

from mail.ipa.ipa.core.context import CoreContext
from mail.ipa.ipa.core.entities.enums import TaskType
from mail.ipa.ipa.core.entities.task import Task
from mail.ipa.ipa.interactions import InteractionClients
from mail.ipa.ipa.storage import Storage, StorageContext
from mail.ipa.ipa.storage.mappers.task import TaskMapper


class BaseAction(BAction):
    context = CoreContext()

    def __init__(self):
        super().__init__()
        self._clients = InteractionClients(self.logger, self.request_id)

    @property
    def clients(self) -> InteractionClients:
        return self._clients

    async def _run(self):
        with self.logger:
            async with self.clients:
                return await super()._run()


class BaseDBAction(BDBAction, BaseAction):
    storage_context_cls = StorageContext

    @property
    def storage(self) -> Storage:  # for autocomplete
        return super().storage


class RunAsyncMixin:
    mapper_name_task: ClassVar[str] = 'task'
    task_type: ClassVar[TaskType]
    logger: LoggerContext

    _init_kwargs: dict
    _init_args: tuple
    context: CoreContext

    def serialize_kwargs(self) -> Dict[str, Any]:
        init_kwargs = dict(**self._init_kwargs)
        annotations = self.__init__.__annotations__  # type: ignore
        assert annotations is not None

        for key in init_kwargs:
            annotation = annotations.get(key)
            if is_dataclass(annotation):
                init_kwargs[key] = asdict(init_kwargs[key])

        return init_kwargs

    @classmethod
    def deserialize_kwargs(cls, params: Dict[str, Any]) -> Dict[str, Any]:
        annotations = cls.__init__.__annotations__  # type: ignore
        assert annotations is not None

        kwargs = {}
        for key, value in params.items():
            param_cls = annotations.get(key)
            if param_cls is None:
                continue

            if is_dataclass(param_cls):
                value = param_cls(**value)

            kwargs[key] = value

        return kwargs

    async def run_async(self, entity_id: Optional[int] = None, meta_info: Optional[Dict[str, Any]] = None) -> Task:
        """
        Creates task instead of running action.
        """
        assert self.context.storage is not None, 'Cannot execute run_async without storage available'
        assert not self._init_args, 'Only kwargs are allowed'
        assert self.mapper_name_task, 'mapper_name_task must be defined'
        assert getattr(self, 'task_type', None), 'task_type must be defined'

        task_mapper: TaskMapper = self.context.storage[self.mapper_name_task]

        init_kwargs = self.serialize_kwargs()

        task = await task_mapper.create(
            task_type=self.task_type,
            entity_id=entity_id,
            params=init_kwargs,
            meta_task_id=self.context.meta_task_id,
            meta_info=meta_info,
        )
        with self.logger:
            self.logger.context_push(task_type=self.task_type.value, entity_id=entity_id, task_id=task.task_id)
            self.logger.info('Created task')
        return task


class BaseRunAsyncDBAction(RunAsyncMixin, BaseDBAction):
    pass


class BaseRunAsyncAction(RunAsyncMixin, BaseAction):
    pass
