from datetime import timedelta
from typing import ClassVar, Tuple

from sendr_utils import utcnow

from mail.ipa.ipa.conf import settings
from mail.ipa.ipa.core.actions.base import BaseDBAction
from mail.ipa.ipa.core.actions.collectors.set_enabled import SetCollectorEnabledAction
from mail.ipa.ipa.core.entities.collector import Collector
from mail.ipa.ipa.core.entities.user import User
from mail.ipa.ipa.interactions.yarm.entities import YarmCollector, YarmCollectorStatus
from mail.ipa.ipa.interactions.yarm.exceptions import YarmNotStartedYetError


class CheckCollectorStatusAction(BaseDBAction):
    ERROR_FLAP_THRESHOLD_MULT: ClassVar[int] = settings.TASKQ_COLLECTOR_ERROR_FLAP_SMOOTH_MULT
    SHUTDOWN_THRESHOLD: ClassVar[timedelta] = timedelta(days=settings.TASKQ_COLLECTOR_SHUTDOWN_THRESHOLD_DAYS)

    def __init__(self, collector: Collector):
        super().__init__()
        self.collector: Collector = collector

    async def _get_yarm_collector(self, collector: Collector, user: User) -> Tuple[YarmCollector, YarmCollectorStatus]:
        assert user.suid and user.uid, 'User should be initialized'
        assert collector.pop_id, 'Collector should be initialized'

        yarm_collector = await self.clients.yarm.get_collector(user.suid, collector.pop_id)
        try:
            # TODO: put status in yarm_collector entity
            yarm_collector_status = await self.clients.yarm.status(collector.pop_id)
        except YarmNotStartedYetError:
            yarm_collector_status = YarmCollectorStatus.get_default_status()

        return yarm_collector, yarm_collector_status

    @classmethod
    def _get_error_status(cls,
                          collector: Collector,
                          yarm_collector: YarmCollector,
                          yarm_collector_status: YarmCollectorStatus) -> str:
        error_status = yarm_collector.error_status
        new_collected = yarm_collector_status.collected
        collected = collector.collected

        if new_collected is not None:
            has_new_collected_letters = collected is None or new_collected > collected
            check_delay = timedelta(minutes=settings.TASKQ_COLLECTOR_CHECK_DELAY_MINUTES)
            modified_recently = utcnow() - collector.modified_at < check_delay * cls.ERROR_FLAP_THRESHOLD_MULT
            if (has_new_collected_letters or modified_recently):
                error_status = Collector.OK_STATUS

        return error_status

    @staticmethod
    def _is_collector_status_changed(collector: Collector,
                                     yarm_collector_status: YarmCollectorStatus,
                                     error_status: str) -> bool:
        return (collector.collected != yarm_collector_status.collected
                or collector.errors != yarm_collector_status.errors
                or collector.total != yarm_collector_status.total
                or collector.status != error_status)

    async def _check_status(self, collector: Collector, user: User) -> Collector:
        self.logger.context_push(prev_collected=collector.collected,
                                 prev_errors=collector.errors,
                                 prev_total=collector.total,
                                 prev_status=collector.status)

        yarm_collector, yarm_collector_status = await self._get_yarm_collector(collector, user)

        error_status = self._get_error_status(collector=collector,
                                              yarm_collector=yarm_collector,
                                              yarm_collector_status=yarm_collector_status)

        self.logger.context_push(collected=yarm_collector_status.collected,
                                 errors=yarm_collector_status.errors,
                                 total=yarm_collector_status.total,
                                 status=error_status,
                                 yarm_error_status=yarm_collector.error_status)

        if self._is_collector_status_changed(collector, yarm_collector_status, error_status):
            # Можно было бы не делать этот IF и обновлять сборщик безусловно.
            # Но это существенная часть логики:
            # Если collected/errors/total/status поменялись - поменяй их и поменяй modified_at.
            # Иначе - не меняй modified_at.

            collector.collected = yarm_collector_status.collected
            collector.errors = yarm_collector_status.errors
            collector.total = yarm_collector_status.total
            collector.status = error_status

            self.logger.info('Collector state updated')
            if collector.is_finished:
                self.logger.info('Collector is finished')

            collector = await self.storage.collector.save(collector)

        return collector

    async def _attempt_shutdown(self, collector: Collector) -> Collector:
        assert collector.collector_id is not None

        if utcnow() - collector.modified_at > self.SHUTDOWN_THRESHOLD and collector.status != collector.OK_STATUS:
            self.logger.info('Shutting down collector')
            return await SetCollectorEnabledAction(collector.collector_id, enabled=False).run()

        return collector

    async def handle(self) -> Collector:
        collector = self.collector
        user = await self.storage.user.get(collector.user_id)
        self.logger.context_push(collector_id=collector.collector_id,
                                 pop_id=collector.pop_id,
                                 user_id=user.user_id,
                                 org_id=user.org_id,
                                 uid=user.uid,
                                 suid=user.suid)

        collector = await self._check_status(collector, user)
        return await self._attempt_shutdown(collector)
