from typing import Optional

from mail.ipa.ipa.core.actions.base import BaseDBAction
from mail.ipa.ipa.core.entities.collector import Collector
from mail.ipa.ipa.core.entities.user import User
from mail.ipa.ipa.core.exceptions import CollectorNotFoundError
from mail.ipa.ipa.interactions.yarm import YarmCollectorStatus
from mail.ipa.ipa.interactions.yarm.exceptions import YarmBaseError
from mail.ipa.ipa.storage.exceptions import CollectorNotFound


class GetCollectorAction(BaseDBAction):
    def __init__(self, collector_id: int, org_id: Optional[int] = None):
        super().__init__()
        self.org_id = org_id
        self.collector_id = collector_id

    async def handle(self) -> Collector:
        self.logger.context_push(
            collector_id=self.collector_id,
            org_id=self.org_id
        )

        try:
            collector: Collector = await self.storage.collector.get(self.collector_id, with_user=True)
            assert isinstance(collector.user, User)
            if self.org_id is not None and collector.user.org_id != self.org_id:
                raise CollectorNotFound
        except CollectorNotFound:
            raise CollectorNotFoundError

        if collector.user.suid is not None and collector.pop_id is not None:
            try:
                collector.yarm_collector = await self.clients.yarm.get_collector(collector.user.suid, collector.pop_id)

                try:
                    collector.yarm_collector.status = await self.clients.yarm.status(collector.pop_id)
                except YarmBaseError as e:
                    self.logger.warning(f'YARM status error: {e.message}')
                    collector.yarm_collector.status = YarmCollectorStatus.get_default_status()
            except YarmBaseError as e:
                self.logger.warning(f'YARM error: {e.message}')

        return collector
