from typing import Dict, List, Optional, cast

from sendr_utils import alist

from mail.ipa.ipa.core.actions.base import BaseDBAction
from mail.ipa.ipa.core.actions.collectors.edit import EditCollectorAction
from mail.ipa.ipa.core.actions.collectors.remove import RemoveCollectorAction
from mail.ipa.ipa.core.entities.collector import Collector
from mail.ipa.ipa.core.entities.import_params import GeneralInitImportParams, ImportParams
from mail.ipa.ipa.core.entities.password import Password
from mail.ipa.ipa.core.entities.user import User
from mail.ipa.ipa.core.exceptions import IpaBaseCoreError, SrcLoginEmptyError, UnknownCollectorFoundError
from mail.ipa.ipa.interactions.directory.entities import DirectoryUser
from mail.ipa.ipa.interactions.yarm.entities import YarmCollector, YarmCollectorState, YarmCollectorStatus
from mail.ipa.ipa.interactions.yarm.exceptions import YarmBaseError, YarmDuplicateError, YarmNotStartedYetError


class InitCollectorAction(BaseDBAction):
    transact = True

    def __init__(self,
                 user: User,
                 directory_user: DirectoryUser,
                 import_params: GeneralInitImportParams,
                 src_login: str,
                 src_password: Password):
        super().__init__()
        self.import_params: GeneralInitImportParams = import_params
        self.user: User = user
        self.directory_user: DirectoryUser = directory_user
        self.src_login: str = src_login
        self.src_password: Password = src_password

    def _get_collector_import_params(self) -> ImportParams:
        general_import_params = self.import_params
        return ImportParams(
            src_login=self.src_login,
            server=general_import_params.server,
            port=general_import_params.port,
            ssl=general_import_params.ssl,
            imap=general_import_params.imap,
            mark_archive_read=general_import_params.mark_archive_read,
            delete_msgs=general_import_params.delete_msgs,
        )

    async def _get_collector_stub(self, persistent_collectors: List[Collector]) -> Collector:
        collector_import_params = self._get_collector_import_params()

        for persistent_collector in persistent_collectors:
            if persistent_collector.pop_id is None and persistent_collector.params == collector_import_params:
                return persistent_collector

        assert self.user.user_id is not None

        collector = Collector(user_id=self.user.user_id, params=collector_import_params)
        collector = await self.storage.collector.create(collector)
        collector.user = self.user

        return collector

    async def _create_collector_in_yarm(self, collector: Collector) -> None:
        assert self.user.suid is not None
        pop_id = await self.clients.yarm.create(src_login=self.src_login,
                                                password=self.src_password.value(),
                                                user_ip=self.import_params.user_ip,
                                                suid=self.user.suid,
                                                dst_email=self.directory_user.email,
                                                params=self.import_params,
                                                )
        collector.pop_id = pop_id
        self.logger.context_push(pop_id=collector.pop_id, collector_id=collector.collector_id)
        await self.storage.collector.save(collector)
        self.logger.info('Created collector')

    async def _on_duplicate_collector(self,
                                      collector: Collector,
                                      persistent_collectors: List[Collector]) -> None:
        # Этот метод вызывается, когда yarm попытался создать сборщик, но не смог из-за своей дедупликации
        # Мы могли бы ничего не делать в этой ситуации. Но сейчас у нас не существует гранулярных механизмов
        # УДАЛЕНИЯ сборщика и РЕДАКТИРОВАНИЯ ПАРОЛЯ сборщика. Поэтому надо попробовать дать пользователю
        # возможность исправить ситуацию. С помощью эвристик
        # Нам надо учитывать следующие замечания
        # Z1. В общем случае у пользователя может быть несколько сборщиков. Поэтому мы не знаем наверняка,
        # какой сборщик конфликтует с создаваемым нами сборщиком (YARM не отдаёт pop_id конфликтующего сборщика).
        # Z2. Редактировать настройки существующего сборщика - нельзя. Можно только пароль. Иначе будут дубли
        # и упячка.
        # Т.о. если хочешь отредактировать server, или port и т.п. => пересоздаёшь сборщик с правильными настройками
        # Исключение - пароль. Пароль редактировать МОЖНО.
        #
        # Итак, две эвристики
        # 1. У нас всего один сборщик, и он находится в ошибочном состоянии. При этом, он не успел собрать ни одного
        # письма. Значит, мы точно знаем, какой сборщик конфликтует (Z1) и мы точно не создадим дубли (Z2).
        # Действие: удалить сборщик. И попытаться создать сборщик заново.
        # 2. Мы нашли сборщик с точно такими же настройками, и он находится в ошибочном состоянии.
        # Тогда попробуем поредачить ему пароль. Даже если пароль при редактировании останется прежним,
        # это действие должно приоретизировать сборщик в YARM - обратная связь будет быстрее.
        user = self.user
        persistent_collectors_map: Dict[str, Collector] = {
            collector.pop_id: collector for collector in persistent_collectors if collector.pop_id is not None
        }

        import_params = self._get_collector_import_params()

        assert user.suid is not None

        yarm_collectors: List[YarmCollector] = await alist(self.clients.yarm.list(suid=user.suid))

        if len(yarm_collectors) == 1:
            yarm_collector = yarm_collectors[0]
            duplicate_collector = persistent_collectors_map.get(yarm_collector.pop_id)
            if duplicate_collector is None:
                raise UnknownCollectorFoundError

            assert duplicate_collector.pop_id

            try:
                status = await self.clients.yarm.status(duplicate_collector.pop_id)
            except YarmNotStartedYetError:
                status = YarmCollectorStatus.get_default_status()

            # Замечание: тут нету условия на yarm_collector.state.
            # Идея следующая: если сборщик сразу перешел в состояние ошибки, то мы сразу отобразим это в интерфейсе.
            # Если при этом сборщик не смог собрать ни одного письма, то почему бы не дать пользователю
            # этот сборщик поредачить?
            if yarm_collector.error_status != YarmCollector.ERROR_STATUS_OK and status.collected is None:
                self.logger.context_push(duplicate_id=duplicate_collector.collector_id,
                                         duplicate_pop_id=duplicate_collector.pop_id,
                                         duplicate_status=duplicate_collector.status)
                self.logger.info('Deleting duplicate malfunctioning collector')
                await RemoveCollectorAction(collector=duplicate_collector).run()
                await self._create_collector_in_yarm(collector)
                return

        for yarm_collector in yarm_collectors:
            pop_id = yarm_collector.pop_id
            persistent_collector = persistent_collectors_map.get(pop_id)
            if persistent_collector is None:
                raise UnknownCollectorFoundError

            if yarm_collector.state == YarmCollectorState.ON:
                continue

            if persistent_collector.params == import_params:
                assert persistent_collector.collector_id is not None
                self.logger.context_push(duplicate_id=persistent_collector.collector_id,
                                         duplicate_pop_id=persistent_collector.pop_id,
                                         duplicate_status=persistent_collector.status)
                self.logger.info('Editing collector')
                await EditCollectorAction(
                    collector_id=persistent_collector.collector_id,
                    password=self.src_password,
                ).run_async(user.user_id)

        # Итак, у нас в базе:
        # 1. Только что создался сборщик
        # 2. ИЛИ уже был сборщик, который не удалось создать ранее из-за ошибки yarm,
        #    у него пустой pop_id, и сейчас мы ретраили его создание
        # Но YARM сказал, что такой сборщик уже есть.
        # Значит, он и в базе у нас уже существует, потому что мы всегда вызываем InheritUserCollectors.
        # Следовательно, сборщик-пустышка из переменной collector в базе не нужен.
        await RemoveCollectorAction(collector=collector).run()

    async def handle(self) -> None:
        user = self.user

        self.logger.context_push(user_id=user.user_id, uid=user.uid, suid=user.suid, org_id=user.org_id)

        assert user.user_id is not None and user.suid is not None
        try:
            if not self.src_login:
                raise SrcLoginEmptyError()

            persistent_collectors = await alist(self.storage.collector.find(user_id=user.user_id))

            collector = await self._get_collector_stub(persistent_collectors)

            try:
                await self._create_collector_in_yarm(collector)
            except YarmDuplicateError:
                self.logger.warning('Collector already exists')
                await self._on_duplicate_collector(collector, persistent_collectors)
        except (YarmBaseError, IpaBaseCoreError) as exc:
            code: Optional[str] = None
            if isinstance(exc, YarmBaseError):
                code = cast(str, exc.CODE)
            elif isinstance(exc, IpaBaseCoreError):
                code = exc.message

            if code is None:
                code = Collector.UNKNOWN_ERROR_STATUS

            with self.logger:
                self.logger.context_push(error_code=code)
                self.logger.warning('Unable to create collector')

            collector.status = code
            await self.storage.collector.save(collector)
            await self.storage.commit()
            raise
