from typing import Optional

from mail.ipa.ipa.core.actions.base import BaseDBAction
from mail.ipa.ipa.core.entities.collector import Collector
from mail.ipa.ipa.core.exceptions import CollectorNotFoundError
from mail.ipa.ipa.storage.exceptions import CollectorNotFound


class RemoveCollectorAction(BaseDBAction):
    transact = True

    def __init__(self,
                 org_id: Optional[int] = None,
                 collector_id: Optional[int] = None,
                 collector: Optional[Collector] = None):
        super().__init__()
        self.org_id = org_id
        self.collector_id = collector_id
        self.collector = collector

    async def _load_collector(self):
        if self.collector is not None:
            self.collector_id = self.collector.collector_id
        elif self.collector_id is not None:
            try:
                self.collector = await self.storage.collector.get(self.collector_id, with_user=True)
                if self.org_id is not None and self.collector.user.org_id != self.org_id:
                    raise CollectorNotFoundError
            except CollectorNotFound:
                raise CollectorNotFoundError
        else:
            raise RuntimeError('No collector or collector_id is provided')

    async def handle(self):
        await self._load_collector()
        assert self.collector and self.collector.user

        with self.logger:
            self.logger.context_push(
                user_id=self.collector.user_id,
                collector_id=self.collector.collector_id,
                pop_id=self.collector.pop_id,
                uid=self.collector.user.uid,
                suid=self.collector.user.suid,
            )

            if self.collector.pop_id:
                await self.clients.yarm.delete_collector(suid=self.collector.user.suid, pop_id=self.collector.pop_id)
            await self.storage.collector.delete(self.collector)
            self.logger.info('Removed collector')
