from typing import Optional

from mail.ipa.ipa.core.actions.base import BaseRunAsyncDBAction
from mail.ipa.ipa.core.entities.collector import Collector
from mail.ipa.ipa.core.entities.enums import TaskType
from mail.ipa.ipa.core.entities.user import User
from mail.ipa.ipa.core.exceptions import CollectorNotFoundError
from mail.ipa.ipa.storage.exceptions import CollectorNotFound


class SetCollectorEnabledAction(BaseRunAsyncDBAction):
    transact = True
    task_type = TaskType.SET_COLLECTOR_ENABLED

    def __init__(self, collector_id: int, enabled: bool, org_id: Optional[int] = None):
        super().__init__()
        self.collector_id: int = collector_id
        self.enabled: bool = enabled
        self.org_id: Optional[int] = org_id

    async def handle(self) -> Collector:
        with self.logger:
            self.logger.context_push(collector_id=self.collector_id, set_enabled=self.enabled, org_id=self.org_id)
            try:
                collector = await self.storage.collector.get(self.collector_id, with_user=True)
                assert isinstance(collector.user, User)
                if self.org_id is not None and collector.user.org_id != self.org_id:
                    raise CollectorNotFound
            except CollectorNotFound:
                raise CollectorNotFoundError

            assert collector.user.suid is not None, 'User suid should be initialized'
            assert collector.pop_id is not None, 'Collector should exists in yarm'

            self.logger.context_push(
                user_id=collector.user.user_id,
                pop_id=collector.pop_id,
                uid=collector.user.uid,
                suid=collector.user.suid
            )

            collector.enabled = self.enabled

            await self.clients.yarm.set_enabled(collector.user.suid, collector.pop_id, self.enabled)
            return await self.storage.collector.save(collector)
