from abc import ABC, abstractmethod
from typing import Any, ClassVar, Dict

from sendr_utils import alist

from mail.ipa.ipa.core.actions.base import BaseDBAction
from mail.ipa.ipa.core.entities.enums import EventType
from mail.ipa.ipa.core.entities.event import Event
from mail.ipa.ipa.core.entities.org import Org
from mail.ipa.ipa.core.entities.task import Task


class BaseCreateImportTaskAction(BaseDBAction, ABC):
    transact = True
    GENERATE_START_EVENT: ClassVar[bool] = False

    def __init__(self, org_id: int):
        super().__init__()
        self.org_id: int = org_id

    @abstractmethod
    async def _handle(self) -> Task:
        pass

    def _get_event_logging_data(self) -> Dict[str, Any]:
        return {}

    async def _generate_event(self, org: Org) -> None:
        await self._generate_start_event(org, self._get_event_logging_data())

    async def _generate_start_event(self, org: Org, data: Dict[str, Any]) -> None:
        assert org.org_id is not None

        # Это явный lock. Если убрать здесь lock, то может получиться гонка.
        org = await self.storage.organization.get(org.org_id, for_update=True)
        assert org.org_id is not None

        events = await alist(self.storage.event.find(org_id=org.org_id,
                                                     event_type=EventType.STOP,
                                                     order_by=('-event_id',),
                                                     limit=1))
        if events:
            stop_event = events[0]
            if stop_event.revision == org.revision:
                org.revision += 1
                await self.storage.organization.save(org)

        await self.storage.event.create(
            Event(
                org_id=org.org_id,
                event_type=EventType.START,
                revision=org.revision,
                data=data,
            )
        )

    async def handle(self) -> int:
        org_id = self.org_id
        # Это get_or_create с lock'ом.
        # Этот lock очень необходим.
        # Пример. Литерой обозначаю идентификатор транзакции
        # 1. A Пришёл /import/{org_id}/stop
        # 2. B Пришёл POST /import/{org_id}
        # 3. A Создали событие STOP, создали задачу на остановку
        # 4. B Выполнили _generate_start_event - номер ревизии не увеличиваем, ведь события STOP мы не видим
        #    (оно пока ещё не закоммичено транзакцией A)
        # 5. B Создали задачу на создание импорта
        # 6. A COMMIT
        # 7. B COMMIT
        # В итоге, задачи выполнятся в порядке A, B. То есть STOP, START. Но у организации не увеличится номер ревизии.
        # Хотя, по логике, должен был (из-за START)
        # Дело в том, что INSERT ON CONFLICT DO NOTHING не берёт lock, если INSERT не произошёл.
        # Поэтому for_update=True необходим
        org = await self.storage.organization.get_or_create(org_id, for_update=True)

        await self._generate_event(org)

        task = await self._handle()
        return task.task_id
