import asyncio
from dataclasses import asdict
from typing import Any, ClassVar, Dict, List

from aiohttp import StreamReader

from sendr_utils import alist

from mail.ipa.ipa.conf import settings
from mail.ipa.ipa.core.actions.base import BaseRunAsyncDBAction
from mail.ipa.ipa.core.actions.import_.base import BaseCreateImportTaskAction
from mail.ipa.ipa.core.actions.import_.org import InitOrgImportAction
from mail.ipa.ipa.core.csvops import CSVEntry, prepare_csv_for_upload, read_downloaded_csv
from mail.ipa.ipa.core.entities.enums import TaskType
from mail.ipa.ipa.core.entities.import_params import GeneralInitImportParams
from mail.ipa.ipa.core.entities.password import Password
from mail.ipa.ipa.core.entities.task import Task
from mail.ipa.ipa.core.entities.user_info import UserInfo
from mail.ipa.ipa.core.exceptions import CSVFieldRequiredError

_password_placeholder = Password('undecipherable')


def _map_csv_entry_to_user_info(entry: CSVEntry, with_password: bool = True) -> UserInfo:
    password = _password_placeholder
    new_password = None
    if with_password:
        password = Password.from_plain(entry[settings.CSV_FIELD_SRC_PASSWORD])
        if entry.get(settings.CSV_FIELD_YANDEXMAIL_PASSWORD):
            new_password = Password.from_plain(entry[settings.CSV_FIELD_YANDEXMAIL_PASSWORD])

    return UserInfo(
        login=entry[settings.CSV_FIELD_YANDEXMAIL_LOGIN],
        password=password,
        src_login=entry[settings.CSV_FIELD_SRC_LOGIN],
        new_password=new_password,
        first_name=entry.get(settings.CSV_FIELD_FIRST_NAME),
        last_name=entry.get(settings.CSV_FIELD_LAST_NAME),
        middle_name=entry.get(settings.CSV_FIELD_MIDDLE_NAME),
        gender=entry.get(settings.CSV_FIELD_GENDER),
        birthday=entry.get(settings.CSV_FIELD_BIRTHDAY),
        language=entry.get(settings.CSV_FIELD_LANGUAGE),
    )


class CreateImportFromCSVAction(BaseCreateImportTaskAction):
    def __init__(self,
                 import_params: GeneralInitImportParams,
                 stream: StreamReader,
                 name: str):
        super().__init__(import_params.org_id)
        self.stream: StreamReader = stream
        self.params: GeneralInitImportParams = import_params
        self.name = name

    @staticmethod
    def validate_csv_entry(entry: CSVEntry, lineno: int) -> None:
        user_info = _map_csv_entry_to_user_info(entry, with_password=False)
        if not user_info.login:
            raise CSVFieldRequiredError(lineno=lineno, field=settings.CSV_FIELD_YANDEXMAIL_LOGIN)
        if not user_info.src_login:
            raise CSVFieldRequiredError(lineno=lineno, field=settings.CSV_FIELD_SRC_LOGIN)
        if not entry[settings.CSV_FIELD_SRC_PASSWORD]:
            raise CSVFieldRequiredError(lineno=lineno, field=settings.CSV_FIELD_SRC_PASSWORD)

    def _get_event_logging_data(self) -> Dict[str, Any]:
        return {'name': self.name, 'params': asdict(self.params)}

    async def _handle(self) -> Task:
        data_iter = prepare_csv_for_upload(self.stream, validate_cb=self.validate_csv_entry)
        # MDS считают, что они не поддерживают Transfer-Encoding: chunked.
        # Технически, такая возможность есть, но, видимо, завязываться на неё нельзя.
        # Поэтому, вместо того, чтобы засылать шифрованный csv чанками,
        # Мы склеиваем все чанки и засылаем в MDS всё тело сразу.
        # Но, в будущем... это можно будет поправить.
        data = b''.join(await alist(data_iter))

        key = await self.clients.mds.upload('csv', data)
        return await ParseCSVAction(
            import_params=self.params,
            csv_key=key,
        ).run_async(entity_id=self.org_id, meta_info={'name': self.name})


class ParseCSVAction(BaseRunAsyncDBAction):
    task_type = TaskType.PARSE_CSV
    transact = True

    PARSE_CSV_LOOP_COOLDOWN: ClassVar[int] = 1000

    def __init__(self, import_params: GeneralInitImportParams, csv_key: str):
        super().__init__()
        self.import_params: GeneralInitImportParams = import_params
        self.csv_key: str = csv_key

    async def handle(self) -> None:
        users: List[UserInfo] = []

        async for entry in read_downloaded_csv(await self.clients.mds.download(self.csv_key)):
            users.append(_map_csv_entry_to_user_info(entry))
            if len(users) % self.PARSE_CSV_LOOP_COOLDOWN == 0:
                await asyncio.sleep(0)

        await InitOrgImportAction(self.import_params, users).run()
