from dataclasses import asdict
from typing import Any, Dict, List

from mail.ipa.ipa.core.actions.base import BaseRunAsyncDBAction
from mail.ipa.ipa.core.actions.import_.user import InitUserImportAction
from mail.ipa.ipa.core.entities.enums import TaskType
from mail.ipa.ipa.core.entities.import_params import GeneralInitImportParams
from mail.ipa.ipa.core.entities.password import Password
from mail.ipa.ipa.core.entities.user import User
from mail.ipa.ipa.core.entities.user_info import UserInfo


class InitOrgImportAction(BaseRunAsyncDBAction):
    task_type = TaskType.INIT_IMPORT
    transact = True

    def __init__(self, params: GeneralInitImportParams, users: List[UserInfo]):
        super().__init__()
        self.params: GeneralInitImportParams = params
        self.users: List[UserInfo] = users

    def serialize_kwargs(self) -> Dict[str, Any]:
        return {
            'params': asdict(self.params),
            'users': [{
                'login': user_info.login,
                'password': user_info.password.encrypted(),
                'src_login': user_info.src_login,
            } for user_info in self.users],
        }

    @classmethod
    def deserialize_kwargs(cls, params: Dict[str, Any]) -> Dict[str, Any]:
        return {
            'params': GeneralInitImportParams(**params['params']),
            'users': [
                UserInfo(
                    login=user_info['login'],
                    password=Password(user_info['password']),
                    src_login=user_info['src_login'],
                )
                for user_info in params['users']
            ],
        }

    @staticmethod
    def _strip_domain(login: str, domains: List[str]) -> str:
        if '@' not in login:
            return login

        local_part, domain = login.rsplit('@', 1)
        if domain in domains:
            return local_part
        return login

    async def handle(self) -> None:
        org_id = self.params.org_id
        directory_org = await self.clients.directory.get_organization(org_id)

        for user in self.users:
            user.login = self._strip_domain(user.login, directory_org.domains.all)
            dbuser = await self.storage.user.get_or_create(User(org_id=org_id, login=user.login))
            assert dbuser.user_id is not None
            await InitUserImportAction(
                import_params=self.params,
                user_info=user,
                user_id=dbuser.user_id,
            ).run_async(dbuser.user_id)
