from typing import ClassVar, List, Optional, Tuple

from sendr_utils import alist

from mail.ipa.ipa.conf import settings
from mail.ipa.ipa.core.actions.base import BaseDBAction
from mail.ipa.ipa.core.entities.collector import Collector
from mail.ipa.ipa.core.entities.enums import UserImportError
from mail.ipa.ipa.core.entities.user import User
from mail.ipa.ipa.utils.helpers import merge_lists


class GetImportInfoAction(BaseDBAction):
    """
    Returns tuples of (user, collector, error) for each user with error and each pair user-collector if user was
    created successfully. If `only_errors` set to True, returns only error users and error user-collector pairs.
    """

    LIMIT: ClassVar[int] = settings.IMPORT_INFO_LIMIT

    def __init__(self, org_id: int, only_errors: bool = False):
        super().__init__()
        self.org_id: int = org_id
        self.only_errors: bool = only_errors

    @staticmethod
    def less_user_collector(user: User, collector: Collector) -> bool:
        assert user.modified_at is not None and collector.modified_at is not None
        return user.modified_at >= collector.modified_at

    async def handle(self) -> Tuple[List[Tuple[User, Optional[Collector], Optional[UserImportError]]], bool]:
        limit_with_extra = self.LIMIT + 1
        users = await alist(self.storage.user.find(
            org_id=self.org_id,
            has_error=True,
            limit=limit_with_extra,
            order_by='modified_at',
            desc=True,
        ))
        collectors = await alist(self.storage.collector.find(
            org_id=self.org_id,
            ok_status=False if self.only_errors else None,
            limit=limit_with_extra,
            order_by='modified_at',
            desc=True,
        ))

        result: List[Tuple[User, Optional[Collector], Optional[UserImportError]]] = []
        for item in merge_lists(users, collectors, self.less_user_collector):
            if isinstance(item, User):
                result.append((item, None, UserImportError.get_error(user_error=item.error, logger=self.logger)))
            elif isinstance(item, Collector):
                assert isinstance(item.user, User)
                result.append((
                    item.user,
                    item,
                    UserImportError.get_error(
                        user_error=item.user.error,
                        collector_status=item.status,
                        logger=self.logger,
                    ),
                ))

        return result[:self.LIMIT], len(result) > self.LIMIT
