import string
from datetime import date, datetime
from typing import Any, Dict, FrozenSet, Optional, Tuple, cast

from sendr_utils import UserType, get_user_type

from mail.ipa.ipa.conf import settings
from mail.ipa.ipa.core.actions.base import BaseDBAction
from mail.ipa.ipa.core.actions.collectors.inherit import InheritUserCollectorsAction
from mail.ipa.ipa.core.actions.users.remove_collectors import RemoveUserCollectorsAction
from mail.ipa.ipa.core.entities.user import User
from mail.ipa.ipa.core.entities.user_info import UserInfo
from mail.ipa.ipa.core.exceptions import (
    InvalidBirthdayError, IpaBaseCoreError, NotConnectUIDError, PasswordSymbolForbiddenError, PasswordWeakError,
    UnsupportedGenderError, UnsupportedLanguageError
)
from mail.ipa.ipa.interactions.blackbox.exceptions import BlackboxBaseError
from mail.ipa.ipa.interactions.directory.entities import DirectoryUser, PasswordMode
from mail.ipa.ipa.interactions.directory.exceptions import DirectoryBaseError, DirectoryNotFoundError
from mail.ipa.ipa.interactions.exceptions import InteractionResponseError


class InitUserAction(BaseDBAction):
    transact = True

    ACCEPTABLE_PASSWORD_SYMBOLS: FrozenSet[str] = frozenset(string.printable)

    def __init__(self, user_id: int, admin_uid: int, user_ip: str, user_info: UserInfo):
        super().__init__()
        self.user_id: int = user_id
        self.admin_uid: int = admin_uid
        self.info: UserInfo = user_info
        self.user_ip: str = user_ip

    @staticmethod
    def _parse_gender(gender: str) -> str:
        if gender not in settings.USER_GENDERS_MAP:
            raise UnsupportedGenderError(gender)
        return settings.USER_GENDERS_MAP[gender]

    @staticmethod
    def _parse_language(language: str) -> str:
        if language not in settings.USER_LANGUAGES_MAP:
            raise UnsupportedLanguageError(language)
        return settings.USER_LANGUAGES_MAP[language]

    @staticmethod
    def _parse_birthday(birthday: str) -> date:
        for fmt in settings.USER_BIRTHDAY_FORMATS:
            try:
                return datetime.strptime(birthday, fmt).date()
            except ValueError:
                pass
        else:
            raise InvalidBirthdayError(birthday)

    def _get_connect_user_params(self) -> Dict[str, Any]:
        data: Dict[str, Any] = {}
        if self.info.first_name is not None:
            data['first_name'] = self.info.first_name
        if self.info.last_name is not None:
            data['last_name'] = self.info.last_name
        if self.info.middle_name is not None:
            data['middle_name'] = self.info.middle_name

        if self.info.birthday is not None:
            data['birthday'] = self._parse_birthday(self.info.birthday)

        if self.info.gender is not None:
            data['gender'] = self._parse_gender(self.info.gender)

        if self.info.language is not None:
            data['language'] = self._parse_language(self.info.language)

        data['password_mode'] = PasswordMode.HASH
        password = self.info.password
        if self.info.new_password is not None:
            password = self.info.new_password

        if not password.value():
            raise PasswordWeakError
        if not set(password.value()).issubset(self.ACCEPTABLE_PASSWORD_SYMBOLS):
            raise PasswordSymbolForbiddenError
        data['password'] = password.md5crypt()

        return data

    async def handle(self) -> Tuple[User, DirectoryUser]:
        user: Optional[User] = None

        try:
            user = await self.storage.user.get(self.user_id)
            self.logger.context_push(user_id=user.user_id, org_id=user.org_id)
            assert user.user_id

            try:
                dir_user = await self.clients.directory.get_user_by_login(user.org_id, user.login)
                self.logger.context_push(uid=dir_user.uid)
            except DirectoryNotFoundError:
                dir_user = await self.clients.directory.create_user(org_id=user.org_id,
                                                                    admin_uid=self.admin_uid,
                                                                    user_ip=self.user_ip,
                                                                    login=user.login,
                                                                    **self._get_connect_user_params(),
                                                                    )
                self.logger.context_push(uid=dir_user.uid)
                self.logger.info('User created')

            # uid у логина в директории отличается от uid в нашей базе
            # Так бывает, если мы ещё не проставили uid пользователю,
            # или если пользователя удалили и снова создали в директории
            if dir_user.uid != user.uid:
                self.logger.context_push(previous_uid=user.uid)
                self.logger.info('Initializing user')
                await RemoveUserCollectorsAction(user.user_id).run()

                new_uid = dir_user.uid
                suid = await self.clients.blackbox.get_suid(new_uid)
                self.logger.context_push(suid=suid)

                user.uid = new_uid
                user.suid = suid
                user.error = None
                await self.storage.user.save(user)

                self.logger.info('User initialized')

            if get_user_type(user.uid) != UserType.CONNECT:
                raise NotConnectUIDError(user.uid)

            await InheritUserCollectorsAction(user).run()

            return user, dir_user
        except (DirectoryBaseError, BlackboxBaseError, IpaBaseCoreError) as exc:
            if isinstance(exc, InteractionResponseError):
                error_code = cast(str, exc.CODE)
            elif isinstance(exc, IpaBaseCoreError):
                error_code = exc.message
            else:
                error_code = None

            if error_code is None:
                error_code = User.UNKNOWN_ERROR

            assert error_code is not None

            self.logger.context_push(error_code=error_code)
            self.logger.warning('Unable to initialize user')

            assert isinstance(user, User) and user.user_id
            user.error = error_code

            await self.storage.user.save(user)
            await RemoveUserCollectorsAction(user.user_id).run()
            await self.storage.commit()

            raise
