import csv
import io
from collections import defaultdict
from typing import AsyncIterable, Callable, ClassVar, Dict, List, Optional

from aiohttp import StreamReader

from sendr_utils import anext

from mail.ipa.ipa.conf import settings
from mail.ipa.ipa.core.crypto import BlockDecryptor, BlockEncryptor
from mail.ipa.ipa.core.csvops.linereader import CSVLineReader, Line
from mail.ipa.ipa.core.entities.csv import CSVParams
from mail.ipa.ipa.core.exceptions import (
    CSVEmptyError, CSVEncodingDetectError, CSVHeaderFieldDuplicateError, CSVHeaderFieldRequiredError,
    CSVHeaderFieldUnknownError, CSVInvalidValueError, CSVMalformedError
)
from mail.ipa.ipa.utils.helpers import aiter

CSVEntry = Dict[str, str]
ValidateCallback = Callable[[CSVEntry, int], None]


class CSVOperations:
    SAMPLE_SIZE = 64 * 1024
    BUFFER_LIMIT = 128 * 1024
    CHUNK_SIZE = 16 * 1024

    def __init__(self, stream: Optional[StreamReader] = None) -> None:
        self.stream: Optional[StreamReader] = stream

    def get_validated_header(self, header: Line, csv_params: CSVParams) -> List[str]:
        try:
            header_reader = csv.DictReader([header.content.decode(csv_params.encoding)], dialect=csv_params.dialect)
        except UnicodeDecodeError:
            raise CSVEncodingDetectError(csv_params.encoding, header.lineno)

        assert header_reader.fieldnames
        fieldnames = list(header_reader.fieldnames)
        headers_map: Dict[str, int] = defaultdict(int)

        for field in fieldnames:
            headers_map[field] += 1
            if field not in settings.CSV_REQUIRED_HEADERS and field not in settings.CSV_OPTIONAL_HEADERS:
                raise CSVHeaderFieldUnknownError(field)
            if headers_map[field] > 1:
                raise CSVHeaderFieldDuplicateError(field)

        for required_header in settings.CSV_REQUIRED_HEADERS:
            if not headers_map[required_header]:
                raise CSVHeaderFieldRequiredError(required_header)

        return fieldnames

    def read_csv_entry(self, line: Line, fieldnames: List[str], csv_params: CSVParams) -> CSVEntry:
        try:
            return next(csv.DictReader([line.content.decode(csv_params.encoding)],
                                       dialect=csv_params.dialect,
                                       fieldnames=fieldnames))
        except UnicodeDecodeError:
            raise CSVEncodingDetectError(csv_params.encoding, line.lineno)


class CSVUploadHelper(CSVOperations):
    MDS_UPLOAD_CHUNK_SIZE: ClassVar[int] = 65536

    def __init__(self,
                 stream: Optional[StreamReader] = None,
                 validate_cb: Optional[ValidateCallback] = None) -> None:
        super().__init__(stream)
        self.validate_cb: Optional[ValidateCallback] = validate_cb

    async def encrypt(self, chunks: AsyncIterable[bytes]) -> AsyncIterable[bytes]:
        enc = BlockEncryptor()

        async for chunk in chunks:
            yield enc.update(chunk)

        yield enc.finalize()

    async def validate(self, line_reader: CSVLineReader) -> AsyncIterable[bytes]:
        params = await line_reader.detect_csv_params()
        lines = aiter(line_reader.readlines())

        header_line = await anext(lines)

        fieldnames = self.get_validated_header(header_line, params)

        yield header_line.content

        entries: int = 0
        async for line in lines:
            entry: CSVEntry = self.read_csv_entry(line, fieldnames, params)
            if None in entry.values() or len(entry) > len(fieldnames):
                raise CSVMalformedError(line.lineno)
            for value in entry.values():
                if '\n' in value or '\r' in value:
                    # Так бывает, если в csv встретилось значение с разделителем строки.
                    # Мы читаем построчно. И фидим csv.reader построчно. А интерфейс у csv.reader
                    # ожидает весь CSV сразу (иными словами, интерфейс не является инкрементальным)
                    # Поэтому, если в значении есть разделитель строки, мы обработаем такие строки некорретно
                    raise CSVInvalidValueError(line.lineno)

            # При импорте CSV из экселя там могут остаться пустые строки, которые попадут в CSV с разделителями
            # Пример: открыл наш шаблон, удалил руками в экселе две строки
            # Решение: игнорируем
            if all(value.strip() == '' for value in entry.values()):
                continue
            if self.validate_cb:
                self.validate_cb(entry, line.lineno)

            entries += 1
            yield line.content

        if entries == 0:
            raise CSVEmptyError

    async def chunk_multiplexer(self, chunks: AsyncIterable[bytes]) -> AsyncIterable[bytes]:
        """
        Зачем нужен chunk_multiplexer:
        TLDR: чтобы загружать (upload) файл большими кусками.
        Метод prepare_for_upload возвращает асинхронный генератор, который предназначен для aiohttp-client.
        Когда aiohttp-client получает генератор, он загружает (upload) файл в режиме Transfer-Encoding: chunked.
        Важная особенность: он нарезает данные на чанки и отправляет их в TCP стек РОВНО ТАК, как их возвращает
        генератор. Это значит, что если в CSV тысяча маленьких строк, то без chunk_multiplexer мы отправим
        в MDS тысячу маленьких чанков, тысячу TCP пакетов, что очень неэффективно.
        """
        buf: io.BytesIO = io.BytesIO()
        async for chunk in chunks:
            buf.write(chunk)
            if buf.tell() >= self.MDS_UPLOAD_CHUNK_SIZE:
                yield buf.getvalue()
                buf.seek(0)
                buf.truncate()
        if buf.tell() > 0:
            yield buf.getvalue()

    def prepare_for_upload(self) -> AsyncIterable[bytes]:
        assert self.stream

        return self.chunk_multiplexer(
            self.encrypt(
                self.validate(
                    CSVLineReader(
                        self.stream.iter_chunked(self.CHUNK_SIZE)
                    )
                )
            )
        )


class CSVDownloadHelper(CSVOperations):
    async def read(self, line_reader: CSVLineReader) -> AsyncIterable[CSVEntry]:
        params = await line_reader.detect_csv_params()
        iter_ = aiter(line_reader.readlines())
        try:
            first_line = await anext(iter_)
        except StopAsyncIteration:
            raise CSVEmptyError

        fieldnames = self.get_validated_header(first_line, params)

        async for line in iter_:
            yield self.read_csv_entry(line, fieldnames, params)

    async def decrypt(self, chunks: AsyncIterable[bytes]) -> AsyncIterable[bytes]:
        dec = BlockDecryptor()
        async for chunk in chunks:
            yield dec.update(chunk)
        yield dec.finalize()

    def read_csv(self) -> AsyncIterable[CSVEntry]:
        assert self.stream

        return self.read(
            CSVLineReader(
                self.decrypt(
                    self.stream.iter_chunked(self.CHUNK_SIZE)
                )
            )
        )


def prepare_csv_for_upload(stream: StreamReader,
                           validate_cb: Optional[ValidateCallback] = None) -> AsyncIterable[bytes]:
    return CSVUploadHelper(stream, validate_cb).prepare_for_upload()


def read_downloaded_csv(stream: StreamReader) -> AsyncIterable[CSVEntry]:
    return CSVDownloadHelper(stream).read_csv()
