from dataclasses import dataclass, field
from datetime import datetime
from typing import ClassVar, Optional, Union

from sendr_utils import utcnow

from mail.ipa.ipa.core.entities.import_params import ImportParams
from mail.ipa.ipa.core.entities.not_fetched import NOT_FETCHED, NotFetchedType
from mail.ipa.ipa.core.entities.user import User
from mail.ipa.ipa.interactions.yarm import YarmCollector


@dataclass
class Collector:
    OK_STATUS: ClassVar[str] = 'ok'
    UNKNOWN_ERROR_STATUS: ClassVar[str] = 'unknown_error'

    user_id: int

    params: ImportParams

    enabled: bool = True
    collector_id: Optional[int] = None
    total: Optional[int] = None
    collected: Optional[int] = None
    errors: Optional[int] = None
    # Сюда пишем ошибку сборщика YARM. Известные статусы: ok | connect error | login error
    status: str = OK_STATUS

    pop_id: Optional[str] = None
    checked_at: Optional[datetime] = field(default_factory=utcnow)
    modified_at: Optional[datetime] = field(default_factory=utcnow)
    created_at: Optional[datetime] = field(default_factory=utcnow)

    # joined
    user: Union[User, NotFetchedType] = NOT_FETCHED
    yarm_collector: Union[YarmCollector, NotFetchedType] = NOT_FETCHED

    @property
    def login(self) -> str:
        assert isinstance(self.user, User)
        return self.user.login

    @property
    def uid(self) -> Optional[int]:
        assert isinstance(self.user, User)
        return self.user.uid

    @property
    def is_ok_status(self) -> bool:
        return self.status == self.OK_STATUS

    @property
    def is_finished(self) -> bool:
        return self.is_ok_status \
            and self.errors is not None \
            and self.collected is not None \
            and self.total is not None \
            and self.errors + self.collected == self.total
