from enum import Enum, unique
from typing import TYPE_CHECKING, ClassVar, Dict, Optional

from sendr_qlog import LoggerContext
from sendr_taskqueue import BaseWorkerType
from sendr_taskqueue.worker.storage import BaseTaskType
from sendr_taskqueue.worker.storage import TaskState as BaseTaskState
from sendr_taskqueue.worker.storage import WorkerState as BaseWorkerState

from mail.ipa.ipa.core.entities.collector import Collector
from mail.ipa.ipa.core.exceptions import (
    InvalidBirthdayError, NotConnectUIDError, PasswordSymbolForbiddenError, PasswordWeakError, SrcLoginEmptyError,
    UnsupportedGenderError, UnsupportedLanguageError
)
from mail.ipa.ipa.interactions.directory.exceptions import (
    DirectoryBirthdayInvalidError, DirectoryLoginLongError, DirectoryLoginProhibitedError, DirectoryPasswordShortError,
    DirectoryPasswordWeakError
)
from mail.ipa.ipa.interactions.yarm.exceptions import (
    YarmAuthError, YarmBlackboxError, YarmCollectFromHimselfError, YarmConnectError, YarmConnectionError,
    YarmLoginError, YarmResolveError, YarmTransportError, YarmUserNotFoundError
)


@unique
class EventType(Enum):
    START = 'start'
    STOP = 'stop'


@unique
class TaskType(BaseTaskType, Enum):
    EDIT_COLLECTOR = 'edit_collector'
    INIT_IMPORT = 'init_import'
    INIT_USER_IMPORT = 'init_user_import'
    PARSE_CSV = 'parse_csv'
    REMOVE_USER_COLLECTORS = 'remove_user_collectors'
    SET_COLLECTOR_ENABLED = 'set_collector_enabled'
    STOP_IMPORT = 'stop_import'


@unique
class WorkerType(BaseWorkerType, Enum):
    ORGANIZATION = 'organization'
    USER = 'user'
    COLLECTOR = 'collector'


@unique
class UserImportError(Enum):
    _ignore_ = ['USER_ERROR_MAP', 'COLLECTOR_STATUS_MAP']
    USER_ERROR_MAP: ClassVar[Dict[str, Optional['UserImportError']]]
    COLLECTOR_STATUS_MAP: ClassVar[Dict[str, Optional['UserImportError']]]

    COLLECTOR_INVALID_CREDENTIALS = 'invalid_credentials'
    COLLECTOR_SERVER_RESOLVE_ERROR = 'server_resolve_error'
    COLLECTOR_TEMPORARY_ERROR = 'collector_temporary_error'
    COLLECTOR_CONNECT_ERROR = 'collector_connect_error'
    COLLECTOR_USER_DELETED = 'user_deleted'
    USER_INVALID_BIRTHDAY = 'invalid_birthday'
    USER_INVALID_GENDER = 'invalid_gender'
    USER_INVALID_LANGUAGE = 'invalid_language'
    USER_LOGIN_LONG = 'login_long'
    USER_LOGIN_PROHIBITED = 'login_prohibited'
    USER_NOT_CONNECT_USER = 'not_connect_user'
    USER_PASSWORD_HAS_FORBIDDEN_SYMBOLS = 'password_has_forbidden_symbols'
    USER_PASSWORD_WEAK = 'password_weak'

    UNKNOWN_ERROR = 'unknown_error'

    @classmethod
    def get_error(cls,
                  user_error: Optional[str] = None,
                  collector_status: Optional[str] = None,
                  logger: Optional[LoggerContext] = None,
                  ) -> Optional['UserImportError']:
        error: 'UserImportError'
        if collector_status is not None:
            error = cls.COLLECTOR_STATUS_MAP.get(collector_status, cls.UNKNOWN_ERROR)  # type: ignore
        else:
            error = cls.USER_ERROR_MAP.get(user_error, cls.UNKNOWN_ERROR)  # type: ignore

        if logger is not None and error == UserImportError.UNKNOWN_ERROR:
            with logger:
                logger.context_push(user_error=user_error, collector_status=collector_status)
                logger.error('Unknown user/collector error')
        return error

    @classmethod
    def get_error_str(cls,
                      user_error: Optional[str] = None,
                      collector_status: Optional[str] = None,
                      logger: Optional[LoggerContext] = None,
                      ) -> str:
        error = cls.get_error(
            user_error=user_error,
            collector_status=collector_status,
            logger=logger,
        )
        return error.value if error else ''


if not TYPE_CHECKING:
    UserImportError.USER_ERROR_MAP = {
        DirectoryBirthdayInvalidError.CODE: UserImportError.USER_INVALID_BIRTHDAY,
        DirectoryLoginLongError.CODE: UserImportError.USER_LOGIN_LONG,
        DirectoryLoginProhibitedError.CODE: UserImportError.USER_LOGIN_PROHIBITED,
        DirectoryPasswordShortError.CODE: UserImportError.USER_PASSWORD_WEAK,
        DirectoryPasswordWeakError.CODE: UserImportError.USER_PASSWORD_WEAK,
        NotConnectUIDError.message: UserImportError.USER_NOT_CONNECT_USER,
        PasswordSymbolForbiddenError.message: UserImportError.USER_PASSWORD_HAS_FORBIDDEN_SYMBOLS,
        PasswordWeakError.message: UserImportError.USER_PASSWORD_WEAK,
        InvalidBirthdayError.message: UserImportError.USER_INVALID_BIRTHDAY,
        UnsupportedGenderError.message: UserImportError.USER_INVALID_GENDER,
        UnsupportedLanguageError.message: UserImportError.USER_INVALID_LANGUAGE,
    }

    UserImportError.COLLECTOR_STATUS_MAP = {
        Collector.OK_STATUS: None,
        SrcLoginEmptyError.message: UserImportError.COLLECTOR_INVALID_CREDENTIALS,
        YarmAuthError.CODE: UserImportError.COLLECTOR_INVALID_CREDENTIALS,
        YarmBlackboxError.CODE: UserImportError.COLLECTOR_TEMPORARY_ERROR,
        YarmConnectError.CODE: UserImportError.COLLECTOR_CONNECT_ERROR,
        YarmConnectionError.CODE: UserImportError.COLLECTOR_CONNECT_ERROR,
        YarmCollectFromHimselfError.CODE: UserImportError.COLLECTOR_INVALID_CREDENTIALS,
        YarmLoginError.CODE: UserImportError.COLLECTOR_INVALID_CREDENTIALS,
        YarmResolveError.CODE: UserImportError.COLLECTOR_SERVER_RESOLVE_ERROR,
        YarmTransportError.CODE: UserImportError.COLLECTOR_TEMPORARY_ERROR,
        YarmUserNotFoundError.CODE: UserImportError.COLLECTOR_USER_DELETED,
    }

TaskState = BaseTaskState
WorkerState = BaseWorkerState
