from typing import Any, Dict, Optional

from sendr_core.exceptions import BaseCoreError, CoreFailError  # noqa


class IpaBaseCoreError(BaseCoreError):
    message = 'CORE_ERROR'


class CoreNotFoundError(IpaBaseCoreError):
    message = 'NOT_FOUND_ERROR'


class UserNotFoundError(CoreNotFoundError):
    message = 'USER_NOT_FOUND_ERROR'


class CollectorNotFoundError(CoreNotFoundError):
    message = 'COLLECTOR_NOT_FOUND_ERROR'


class AuthorizationError(IpaBaseCoreError):
    message = 'AUTHORIZATION_ERROR'


class CoreBadRequestError(IpaBaseCoreError):
    message = 'BAD_REQUEST'


class ImportEmptyError(CoreBadRequestError):
    message = 'IMPORT_EMPTY'


class CSVError(CoreBadRequestError):
    message = 'CSV_ERROR'
    params: Dict[str, Any]

    def __init__(self):
        super().__init__()
        self.params = {}


class CSVEmptyError(CSVError):
    message = 'CSV_EMPTY'


class CSVHeaderError(CSVError):
    def __init__(self, field: str) -> None:
        super().__init__()
        self.params = {
            'field_name': field,
        }


class CSVHeaderFieldRequiredError(CSVHeaderError):
    message = 'CSV_HEADER_FIELD_REQUIRED'


class CSVHeaderFieldUnknownError(CSVHeaderError):
    message = 'CSV_HEADER_FIELD_UNKNOWN'


class CSVHeaderFieldDuplicateError(CSVHeaderError):
    message = 'CSV_HEADER_FIELD_DUPLICATE'


class CSVIsTooBig(CSVError):
    message = 'CSV_TOO_BIG'

    def __init__(self, max_size: int) -> None:
        super().__init__()
        self.params = {
            'max_size_bytes': max_size,
        }


class CSVLineIsTooBig(CSVIsTooBig):
    message = 'CSV_LINE_TOO_BIG'

    def __init__(self, max_size: int, lineno: int):
        super().__init__(max_size)
        self.params['lineno'] = lineno


class CSVMalformedError(CSVError):
    message = 'CSV_MALFORMED'

    def __init__(self, lineno: int) -> None:
        super().__init__()
        self.params = {
            'lineno': lineno,
        }


class CSVFieldRequiredError(CSVMalformedError):
    def __init__(self, lineno: int, field: str) -> None:
        super().__init__(lineno)
        self.params['field_name'] = field


class CSVInvalidValueError(CSVMalformedError):
    message = 'CSV_INVALID_VALUE'

    def __init__(self, lineno: int):
        super().__init__(lineno)


class CSVEncodingDetectError(CSVError):
    message = 'CSV_AMBIGOUS_ENCODING'

    def __init__(self, encoding: str, line: Optional[int] = None, confidence: Optional[float] = None):
        super().__init__()
        self.params: Dict[str, Any] = {
            'encoding': encoding,
        }

        if line is not None:
            self.params['line'] = line

        if line is not None:
            self.params['confidence'] = confidence


class UnknownCollectorFoundError(IpaBaseCoreError):
    """Кто-то создаёт сборщики одновременно с нашим бэкендом. Такое лучше поретраить."""


class NotConnectUIDError(IpaBaseCoreError):
    message = 'NOT_CONNECT_USER'

    def __init__(self, uid: int):
        self.params = {'uid': uid}


class InvalidBirthdayError(IpaBaseCoreError):
    message = 'INVALID_BIRTHDAY'

    def __init__(self, datestr: str):
        super().__init__()
        self.params = {
            'datestr': datestr,
        }


class UnsupportedGenderError(IpaBaseCoreError):
    message = 'INVALID_GENDER'

    def __init__(self, gender: str):
        super().__init__()
        self.params = {
            'gender': gender,
        }


class UnsupportedLanguageError(IpaBaseCoreError):
    message = 'INVALID_LANGUAGE'

    def __init__(self, language: str):
        super().__init__()
        self.params = {
            'language': language,
        }


class PasswordInvalidError(IpaBaseCoreError):
    """
    Мы демократично относимся к паролям пользователей. Но есть разумные ограничения, без которых никак.
    """


class PasswordWeakError(PasswordInvalidError):
    """
    Пустой пароль - слабый пароль. А ещё по нему нельзя залогиниться в паспорте.
    """
    message = 'PASSWORD_WEAK'


class PasswordSymbolForbiddenError(PasswordInvalidError):
    """
    Мы демократично относимся к паролям пользователей. Но есть разумные ограничения, без которых никак.
    Пароль содержит недопустимые символы. С таким паролем, скорее всего, не получится залогиниться в паспорте.
    """
    message = 'PASSWORD_HAS_FORBIDDEN_SYMBOLS'


class SrcLoginEmptyError(IpaBaseCoreError):
    message = 'SRC_LOGIN_IS_EMPTY'
