import ssl
from typing import Optional, TypeVar

from aiohttp import TCPConnector  # type: ignore

from sendr_interactions import AbstractInteractionClient
from sendr_tvm import qloud_async_tvm as qtvm
from sendr_tvm.client import aiohttp as tvm_session

from mail.ipa.ipa.conf import settings
from mail.ipa.ipa.utils.stats import (
    interaction_method_response_status, interaction_method_response_time, interaction_response_status,
    interaction_response_time
)
from mail.ipa.ipa.utils.tvm import TVM_CONFIG

SSL_CONTEXT = ssl.create_default_context()
if settings.CA_FILE is not None:
    SSL_CONTEXT.load_verify_locations(cafile=settings.CA_FILE)


def create_connector():
    return TCPConnector(
        keepalive_timeout=settings.KEEPALIVE_TIMEOUT,
        limit=settings.CONNECTION_LIMIT,
        ssl=SSL_CONTEXT,
    )


IpaClientSession = tvm_session.sessions_producer(lambda: qtvm.QTVM(**TVM_CONFIG))

ResponseType = TypeVar('ResponseType')


class BaseInteractionClient(AbstractInteractionClient[ResponseType]):
    DEBUG = settings.DEBUG
    REQUEST_RETRY_TIMEOUTS = settings.REQUEST_RETRY_TIMEOUTS
    TVM_SESSION_CLS = IpaClientSession

    @classmethod
    def _response_time_metrics(cls, interaction_method: str, response_time: int) -> None:
        interaction_response_time.labels(cls.SERVICE).observe(response_time)
        interaction_method_response_time.labels(cls.SERVICE, interaction_method).observe(response_time)

    @classmethod
    def _response_status_metrics(cls, interaction_method: str, status: int) -> None:
        short_status = str(status // 100) + 'xx'
        interaction_response_status.labels(cls.SERVICE, short_status).inc()
        interaction_method_response_status.labels(cls.SERVICE, interaction_method, short_status).inc()

    def endpoint_url(self, relative_url: str, base_url_override: Optional[str] = None) -> str:
        base_url = base_url_override or self.BASE_URL
        base_url = base_url.rstrip('/')
        relative_url = relative_url.lstrip('/')
        return f'{base_url}/{relative_url}'

    def _get_session_kwargs(self):
        kw = super()._get_session_kwargs()
        kw['trust_env'] = True
        return kw

    @classmethod
    async def close_connector(cls):
        if cls.CONNECTOR:
            await cls.CONNECTOR.close()
            cls.CONNECTOR = None
