from aiohttp import ClientResponse

from mail.ipa.ipa.conf import settings
from mail.ipa.ipa.interactions.base import BaseInteractionClient
from mail.ipa.ipa.interactions.blackbox.exceptions import BlackboxBaseError, BlackboxUserNotFoundError


class BlackboxClient(BaseInteractionClient[dict]):
    SERVICE = 'blackbox'
    BASE_URL = settings.BLACKBOX_API_URL
    TVM_ID = settings.BLACKBOX_TVM_ID

    async def _process_response(self, response: ClientResponse, interaction_method: str) -> dict:
        if response.status >= 400:
            await self._handle_response_error(response)
        response_data = await response.json()
        if 'error' in response_data:
            await self._handle_response_error(response)
        return response_data

    async def _handle_response_error(self, response: ClientResponse) -> None:
        if response.status >= 400:
            await super()._handle_response_error(response)
        response_data = await response.json()
        code = response_data.get('error')
        exc = BlackboxBaseError.get_exception_by_code(code)
        raise exc(response.status,
                  code=code,
                  service=self.SERVICE,
                  method=response.method)

    async def get_suid(self, uid: int) -> int:
        response = await self.get('get_suid',
                                  self.BASE_URL,
                                  params={
                                      'method': 'userinfo',
                                      'uid': uid,
                                      'userip': '127.0.0.1',
                                      'dbfields': 'subscription.suid.2',
                                      'format': 'json',
                                  }
                                  )

        user = response['users'][0]
        suid = user.get('dbfields', {}).get('subscription.suid.2')
        if not suid:
            raise BlackboxUserNotFoundError(200, service=self.SERVICE, method='get_suid')
        return suid
