from datetime import date
from typing import Optional
from urllib.parse import quote

from aiohttp import ClientResponse

from mail.ipa.ipa.conf import settings
from mail.ipa.ipa.interactions.base import BaseInteractionClient
from mail.ipa.ipa.interactions.directory.entities import DirectoryOrg, DirectoryUser, PasswordMode
from mail.ipa.ipa.interactions.directory.exceptions import DirectoryBaseError


class DirectoryClient(BaseInteractionClient[dict]):
    SERVICE = 'directory'
    BASE_URL = settings.DIRECTORY_API_URL
    TVM_ID = settings.DIRECTORY_TVM_ID

    LOGGING_SENSITIVE_FIELDS = ('password', 'password_hash')
    LOGGING_EXPOSED_HEADERS = ('x-org-id', 'x-uid', 'x-user-ip')

    async def _handle_response_error(self, response: ClientResponse) -> None:
        response_data = await response.json()
        error_code = response_data.get('code')
        exc = DirectoryBaseError.get_exception_by_code(error_code)
        raise exc(response.status,
                  code=error_code,
                  service=self.SERVICE,
                  method=response.method)

    async def create_user(self,
                          org_id: int,
                          admin_uid: int,
                          user_ip: str,
                          login: str,
                          password: str,
                          password_mode: PasswordMode = PasswordMode.PLAIN,
                          first_name: Optional[str] = None,
                          last_name: Optional[str] = None,
                          middle_name: Optional[str] = None,
                          gender: Optional[str] = None,
                          language: Optional[str] = None,
                          birthday: Optional[date] = None,
                          ) -> DirectoryUser:

        url = self.endpoint_url('v11/users/')

        data = {
            'nickname': login,
            'department_id': 1,
            'password': password,
            'password_mode': password_mode.value,
        }
        if birthday is not None:
            data['birthday'] = birthday.isoformat()
        if gender is not None:
            data['gender'] = gender
        if language is not None:
            data['language'] = language

        if first_name is not None or last_name is not None or middle_name is not None:
            data['name'] = {}
            if first_name is not None:
                data['name']['first'] = {
                    'ru': first_name,
                    'en': first_name,
                }
            if last_name is not None:
                data['name']['last'] = {
                    'ru': last_name,
                    'en': last_name,
                }

            if middle_name is not None:
                data['name']['middle'] = {
                    'ru': middle_name,
                    'en': middle_name,
                }

        response = await self.post(
            'create_user',
            url,
            json=data,
            headers={
                'X-ORG-ID': str(org_id),
                'X-UID': str(admin_uid),
                'X-USER-IP': user_ip,
            },
        )

        return DirectoryUser.from_directory_response(response)

    async def get_user_by_login(self, org_id: int, nickname: str) -> DirectoryUser:
        url = self.endpoint_url(f'v11/users/nickname/{quote(nickname)}/')

        response = await self.get(
            'get_user_by_login',
            url,
            params={
                'fields': 'id,email',
            },
            headers={
                'X-ORG-ID': str(org_id),
            },
        )
        return DirectoryUser.from_directory_response(response)

    async def get_organization(self, org_id: int) -> DirectoryOrg:
        raw_org_id = str(int(org_id))
        url = self.endpoint_url(f'v11/organizations/{raw_org_id}/')

        response = await self.get(
            'get_organization',
            url,
            params={
                'fields': 'id,domains',
            },
            headers={
                'X-ORG-ID': raw_org_id,
            },
        )
        return DirectoryOrg.from_directory_response(response)
