from typing import ClassVar, Dict, Generic, Optional, Type, TypeVar

from sendr_interactions.exceptions import InteractionResponseError as BaseInteractionResponseError

ErrorCodeType = TypeVar('ErrorCodeType')


class InteractionResponseError(Generic[ErrorCodeType], BaseInteractionResponseError):
    CODE: Optional[ErrorCodeType] = None

    _exceptions_by_code: ClassVar[Dict[ErrorCodeType, Type['InteractionResponseError']]]

    def __init_subclass__(cls) -> None:
        code = getattr(cls, 'CODE', None)
        if code is not None:
            assert code not in cls._exceptions_by_code
            cls._exceptions_by_code[code] = cls

    @classmethod
    def get_exception_by_code(cls, code: ErrorCodeType) -> Type['InteractionResponseError']:
        return cls._exceptions_by_code.get(code, cls)

    def __init__(self,
                 status: int,
                 *,
                 code: Optional[ErrorCodeType] = None,
                 service: str,
                 method: str,
                 message: Optional[str] = None):
        super().__init__(
            status_code=status,
            service=service,
            method=method,
            message=message,
        )
        if self.CODE is None:
            self.CODE = code
