import uuid
from typing import ClassVar, Optional

import xmltodict
from aiohttp import ClientResponse, StreamReader

from sendr_utils.retry import async_retry

from mail.ipa.ipa.conf import settings
from mail.ipa.ipa.interactions.base import BaseInteractionClient
from mail.ipa.ipa.interactions.mds.exceptions import MDSBaseError, MDSForbiddenError


class MDSClient(BaseInteractionClient[ClientResponse]):
    SERVICE: ClassVar[str] = 'mds'
    TVM_ID: ClassVar[int] = settings.MDS_TVM_ID

    READ_BASE_URL: ClassVar[str] = settings.MDS_READ_URL
    WRITE_BASE_URL: ClassVar[str] = settings.MDS_WRITE_URL
    NAMESPACE: ClassVar[str] = settings.MDS_NAMESPACE
    EXPIRE: ClassVar[Optional[str]] = settings.MDS_EXPIRE

    async def _process_response(self, response: ClientResponse, interaction_method: str) -> ClientResponse:
        if response.status >= 400:
            await self._handle_response_error(response)
        return response

    async def _handle_response_error(self, response: ClientResponse) -> None:
        code = response.status
        exc = MDSBaseError.get_exception_by_code(code)  # type: ignore
        raise exc(response.status,
                  code=code,
                  service=self.SERVICE,
                  method=response.method)

    def write_url(self, relative_url: str) -> str:
        return self.endpoint_url(relative_url, base_url_override=self.WRITE_BASE_URL)

    def read_url(self, relative_url: str) -> str:
        return self.endpoint_url(relative_url, base_url_override=self.READ_BASE_URL)

    @async_retry(
        base_delay=settings.MDS_UPLOAD_RETRY_BASE_DELAY,
        retries=settings.MDS_UPLOAD_RETRIES,
        exceptions=(MDSForbiddenError,),
        wrap_exception=False,
    )
    async def upload(self, filename: str, data: bytes) -> str:
        url = self.write_url(f'upload-{self.NAMESPACE}/{filename}.{uuid.uuid4().hex}')
        assert self.EXPIRE
        assert isinstance(data, bytes)
        params = {'expire': self.EXPIRE}
        response = await self.post(
            'upload',
            url,
            data=data,
            params=params,
        )
        response_data = xmltodict.parse(await response.read(), dict_constructor=dict)
        return response_data['post']['@key']

    def download_url(self, filepath: str) -> str:
        return self.read_url(f'get-{self.NAMESPACE}/{filepath}')

    async def download(self, filepath: str) -> StreamReader:
        """
        Can be used with web.StreamResponse to stream response inside handler.
        To do this `close` must be set to True so that session is closed once data is read.
        """
        url = self.download_url(filepath)
        response = await self.get('download', url)
        return response.content
