from typing import Any, AsyncGenerator, List

from aiohttp import ClientResponse

from mail.ipa.ipa.conf import settings
from mail.ipa.ipa.core.entities.import_params import GeneralImportParams
from mail.ipa.ipa.interactions.base import BaseInteractionClient
from mail.ipa.ipa.interactions.yarm.entities import YarmCollector, YarmCollectorFolderStatus, YarmCollectorStatus
from mail.ipa.ipa.interactions.yarm.exceptions import YarmBaseError


class YarmClient(BaseInteractionClient[dict]):
    SERVICE = 'yarm'
    BASE_URL = settings.YARM_API_URL

    LOGGING_SENSITIVE_FIELDS = ('password',)
    LOGGING_EXPOSED_HEADERS = ('x-real-ip',)

    async def _process_response(self, response: ClientResponse, interaction_method: str) -> dict:
        if response.status >= 400:
            await self._handle_response_error(response)
        response_data = await response.json()
        if 'error' in response_data:
            await self._handle_response_error(response)
        return response_data

    async def _handle_response_error(self, response: ClientResponse) -> None:
        if response.status >= 400:
            await super()._handle_response_error(response)
        response_data = await response.json()

        error = response_data.get('error')
        assert error, 'Response expected to contain error, but successful response found'

        error_code = error.get('reason')
        error_description = error.get('description')
        exc = YarmBaseError.get_exception_by_code(error_code)
        raise exc(response.status,
                  code=error_code,
                  message=error_description,
                  service=self.SERVICE,
                  method=response.method)

    async def _make_request(self, *args: Any, **kwargs: Any) -> ClientResponse:
        kwargs.setdefault('params', {})
        kwargs['params']['json'] = 1
        return await super()._make_request(*args, **kwargs)

    async def create(self,
                     src_login: str,
                     password: str,
                     suid: int,
                     user_ip: str,
                     dst_email: str,
                     params: GeneralImportParams) -> str:
        """
        Создаёт сборщик.
        Возвращает popid - идентификатор сборщика.
        Если похожий сборщик уже существует, выбросится "DuplicateError".

        dst_email - нужен для дедупликации. Вообще, туда можно передать что угодно.
                    Ящик назначения определяется по SUID.
        """
        url = self.endpoint_url('api/create')
        response = await self.post('create',
                                   url,
                                   params={
                                       'mdb': 'pg',
                                       'login': src_login,
                                       'suid': suid,
                                       'user': dst_email,
                                       'server': params.server,
                                       'port': params.port,
                                       'ssl': int(params.ssl),
                                       'imap': int(params.imap),
                                       'mark_archive_read': int(params.mark_archive_read),
                                       'no_delete_msgs': int(not params.delete_msgs),
                                   },
                                   data={
                                       'password': password,
                                   },
                                   headers={
                                       'x-real-ip': user_ip,
                                   },
                                   )
        return response['popid']

    async def list(self, suid: int) -> AsyncGenerator[YarmCollector, None]:
        url = self.endpoint_url('api/list')
        response = await self.get('list',
                                  url,
                                  params={
                                      'mdb': 'pg',
                                      'suid': suid,
                                  },
                                  )
        for collector in response['rpops']:
            yield YarmCollector.from_response(collector)

    async def set_enabled(self, suid: int, pop_id: str, enabled: bool) -> None:
        url = self.endpoint_url('api/enable')
        await self.post('set_enabled',
                        url,
                        params={
                            'mdb': 'pg',
                            'suid': suid,
                            'popid': pop_id,
                            'is_on': int(enabled)
                        },
                        )

    async def get_collector(self, suid: int, pop_id: str) -> YarmCollector:
        url = self.endpoint_url('api/list')
        response = await self.get('get_collector',
                                  url,
                                  params={
                                      'mdb': 'pg',
                                      'suid': suid,
                                      'popid': pop_id,
                                  },
                                  )

        rpops = response['rpops']
        assert len(rpops) == 1  # yarm возвращает "not found", если сборщик не найден
        return YarmCollector.from_response(rpops[0])

    async def status(self, pop_id: str) -> YarmCollectorStatus:
        url = self.endpoint_url('api/status')
        response = await self.get('get_status',
                                  url,
                                  params={
                                      'popid': pop_id,
                                  },
                                  )
        total, collected, errors = 0, 0, 0
        folders: List[YarmCollectorFolderStatus] = []
        for folder in response['folders']:
            status = YarmCollectorFolderStatus(
                name=folder['name'],
                total=int(folder['messages']),
                errors=int(folder['errors']),
                collected=int(folder['collected'])
            )

            total += status.total
            errors += status.errors
            collected += status.collected

            folders.append(status)

        return YarmCollectorStatus(total=total, collected=collected, errors=errors, folders=folders)

    async def edit(self,
                   suid: int,
                   pop_id: str,
                   dst_email: str,
                   password: str) -> None:
        url = self.endpoint_url('api/edit')
        await self.post('edit',
                        url,
                        params={
                            'mdb': 'pg',
                            'popid': pop_id,
                            'suid': suid,
                            'user': dst_email,
                        },
                        data={
                            'password': password,
                        }
                        )

    async def delete_collector(self, suid: int, pop_id: str) -> None:
        url = self.endpoint_url('api/delete')
        await self.post('delete',
                        url,
                        params={
                            'mdb': 'pg',
                            'popid': pop_id,
                            'suid': suid,
                        },
                        )
