from dataclasses import dataclass, field
from enum import Enum, unique
from typing import Any, Dict, List, Optional, Union

from mail.ipa.ipa.core.entities.not_fetched import NOT_FETCHED, NotFetchedType


@unique
class YarmCollectorState(Enum):
    """
    Работает ли сборщик
    """
    OFF = 0
    ON = 1
    TEMPORARY_ERROR = 2
    PERSISTENT_ERROR = 3
    UNKNOWN = 4

    @classmethod
    def from_yarm_is_on(cls, is_on: int) -> 'YarmCollectorState':
        try:
            return cls(is_on)
        except ValueError:
            return cls.UNKNOWN


@dataclass
class YarmCollectorFolderStatus:
    name: str
    collected: int
    errors: int
    total: int


@dataclass
class YarmCollectorStatus:
    """
    Статусы по собранным письмам
    """
    collected: Optional[int]
    errors: Optional[int]
    total: Optional[int]

    folders: List[YarmCollectorFolderStatus] = field(default_factory=list)

    @classmethod
    def get_default_status(cls):
        return cls(collected=None, errors=None, total=None)


@dataclass
class YarmCollector:
    """
    Строится из вот такой JSON'ины.
    {
      "popid": "2044",
      "server": "imap.yandex.ru",
      "port": "993",
      "login": "rpop-test1@example.test",
      "use_ssl": true,
      "email": "rpop-test1@example.test",
      "is_on": "1",
      "last_connect": "1576342946",
      "session_duration": "2",
      "bad_retries": 0,
      "error_status": "ok",
      "last_msg_count": 0,
      "leave_msgs": false,
      "abook_sync_state": "0",
      "imap": true,
      "root_folder": "",
      "label_id": "",
      "is_oauth": false,
      "actions": [
        "0"
      ]
    }
    """
    ERROR_STATUS_OK = 'ok'

    pop_id: str
    server: str
    port: int
    login: str
    ssl: bool
    email: str
    imap: bool
    state: YarmCollectorState
    delete_msgs: bool
    error_status: str = ERROR_STATUS_OK

    # joined
    status: Union[YarmCollectorStatus, NotFetchedType] = NOT_FETCHED

    @classmethod
    def from_response(cls, response: Dict[str, Any]) -> 'YarmCollector':
        error_status = response['error_status']

        # Пустая строка может проскакивать в ошибках. Пока не понятно, почему.
        # https://st.yandex-team.ru/RTEC-3912#5e58cd8f58bb4b2c058cf75d
        if error_status == '':
            error_status = cls.ERROR_STATUS_OK
        return cls(
            pop_id=response['popid'],
            server=response['server'],
            error_status=error_status,
            port=int(response['port']),
            login=response['login'],
            ssl=response['use_ssl'],
            email=response['email'],
            imap=response['imap'],
            delete_msgs=not response['leave_msgs'],
            state=YarmCollectorState.from_yarm_is_on(int(response['is_on'])),
        )
