from typing import ClassVar, Dict, Type

from mail.ipa.ipa.interactions.exceptions import InteractionResponseError


class YarmBaseError(InteractionResponseError[str]):
    """
    https://wiki.yandex-team.ru/Pochta/collectors/api/#otvetoboshibke
    https://a.yandex-team.ru/arc/trunk/arcadia/mail/rpop

    Пример ответа об ошибке:
    {
      "error": {
        "description": "passport_error",
        "host": "iva3-21539a779b83.qloud-c.yandex.net",
        "request_id": "wFST8tu5tSw1",
        "method": "/api/list",
        "reason": "passport_error"
      }
    }
    """

    _exceptions_by_code: ClassVar[Dict[str, Type[InteractionResponseError]]] = {}

    @classmethod
    def get_exception_by_code(cls, code: str) -> Type['InteractionResponseError']:
        if code is not None:
            return cls._exceptions_by_code.get(code.replace(' ', '_'), cls)
        return cls


class YarmDuplicateError(YarmBaseError):
    """
    Такой сборщик уже есть
    """
    CODE = 'dublicate_error'


class YarmConnectError(YarmBaseError):
    """
    Не удалось подключиться к POP3/IMAP серверу
    """
    CODE = 'connect_error'


class YarmConnectionError(YarmConnectError):
    """
    Тоже самое, но иначе называется.
    (возникает в другой момент процесса работы сборщика, нежели предыдущая ошибка)
    """
    CODE = 'connection_error'


class YarmResolveError(YarmBaseError):
    """
    DNS запись src сервера не удалось разрезолвить
    """
    CODE = 'resolve_error'


class YarmLoginError(YarmBaseError):
    """
    Неправильные реквизиты доступа (login/password)
    """
    CODE = 'login_error'


class YarmNotFoundError(YarmBaseError):
    """
    Метод: list. Сборщик не найден.
    """
    CODE = 'not_found'


class YarmNotStartedYetError(YarmBaseError):
    """
    Метод: status. Сборщик либо не существует, либо не совершил ни одной сессии сбора.
    """
    CODE = 'collector_not_started_yet'


class YarmTransportError(YarmBaseError):
    """
    Метод: list.
    Во время последней сессии сбора были сетевые проблемы (например, порвалось соединение с IMAP сервером).
    При этом предыдущие сессии вполне могли завершаться успешно.
    """
    CODE = 'transport_error'


class YarmAuthError(YarmBaseError):
    """
    Метод: list. Видимо, сочетает в себе YarmConnectError и YarmLoginError.
    Из метода list эти две вышеупомянутые ошибки не прилетают вообще, к сожалению.
    """
    CODE = 'auth_error'


class YarmBlackboxError(YarmBaseError):
    """
    YARM получил какое-то исключение при попытке сходить в Blackbox.
    """
    CODE = 'blackbox_error'


class YarmUserNotFoundError(YarmBaseError):
    """
    Blackbox говорит YARM'у, что пользователя не существует.
    """
    CODE = 'nonexistent_user'


class YarmCollectFromHimselfError(YarmBaseError):
    """
    Нельзя импортировать из себя в себя.
    """
    CODE = 'rpop_from_himself_error'
