from sqlalchemy import and_, case, func, join, select

from mail.ipa.ipa.core.entities.collector import Collector
from mail.ipa.ipa.core.entities.stat import CollectorStat, UserStat
from mail.ipa.ipa.storage.db.tables import collectors as t_collectors
from mail.ipa.ipa.storage.db.tables import users as t_users
from mail.ipa.ipa.storage.gateways.base import BaseGateway


class StatGateway(BaseGateway):
    table = t_users

    async def get_user_stat(self, org_id: int) -> UserStat:
        query = (
            select([func.count()]).
            select_from(t_users).
            where(t_users.c.org_id == org_id).
            where(t_users.c.error != None)  # noqa
        )
        row = await self._query_one(query)
        return UserStat(errors=row[0])

    async def get_collector_stat(self, org_id: int) -> CollectorStat:
        query = (
            select([
                func.count(),
                func.sum(case([
                    (t_collectors.c.status != Collector.OK_STATUS, 1),
                ])),
                func.sum(case([
                    (
                        and_(
                            t_collectors.c.status == Collector.OK_STATUS,
                            t_collectors.c.collected + t_collectors.c.errors == t_collectors.c.total,
                        ),
                        1
                    ),
                ]))
            ]).
            select_from(join(t_users, t_collectors, t_users.c.user_id == t_collectors.c.user_id)).
            where(t_users.c.org_id == org_id)
        )
        row = await self._query_one(query)
        return CollectorStat(
            total=row[0],
            errors=row[1] or 0,  # func.sum returns None in case no matching rows found
            finished=row[2] or 0,
        )
