from datetime import datetime
from typing import AsyncIterable, Optional, Tuple

from sendr_aiopg import BaseMapperCRUD, CRUDQueries
from sendr_aiopg.data_mapper import SelectableDataMapper, TableDataDumper
from sendr_aiopg.query_builder import Filters

from mail.ipa.ipa.core.entities.enums import EventType
from mail.ipa.ipa.core.entities.event import Event
from mail.ipa.ipa.storage.db.tables import events as t_events


class EventDataMapper(SelectableDataMapper):
    entity_class = Event
    selectable = t_events


class EventDataDumper(TableDataDumper):
    entity_class = Event
    table = t_events


class EventMapper(BaseMapperCRUD[Event]):
    name = 'event'
    model = Event

    _builder = CRUDQueries(
        base=t_events,
        id_fields=('event_id',),
        mapper_cls=EventDataMapper,
        dumper_cls=EventDataDumper,
    )

    async def create(self, item: Event) -> Event:  # type: ignore
        return await super().create(item, ignore_fields=self._builder.id_fields)

    def find(self,  # type: ignore
             org_id: int,
             event_type: Optional[EventType] = None,
             order_by: Optional[Tuple[str, ...]] = None,
             offset: Optional[int] = None,
             created_at_from: Optional[datetime] = None,
             created_at_to: Optional[datetime] = None,
             limit: Optional[int] = None,
             iterator: bool = False,
             ) -> AsyncIterable[Event]:
        filters = Filters()
        filters.add_not_none('org_id', org_id)
        filters.add_not_none('created_at', created_at_from, lambda field: created_at_from <= field)
        filters.add_not_none('created_at', created_at_to, lambda field: field < created_at_to)
        filters.add_not_none('event_type', event_type)

        return super().find(filters=filters, order=order_by, offset=offset, limit=limit)
