from sqlalchemy import select
from sqlalchemy.dialects.postgresql import insert

from sendr_aiopg import BaseMapperCRUD, CRUDQueries
from sendr_aiopg.data_mapper import SelectableDataMapper, TableDataDumper

from mail.ipa.ipa.core.entities.org import Org
from mail.ipa.ipa.storage.db.tables import organizations as t_organizations


class OrgDataMapper(SelectableDataMapper):
    entity_class = Org
    selectable = t_organizations


class OrgDataDumper(TableDataDumper):
    entity_class = Org
    table = t_organizations


class OrganizationMapper(BaseMapperCRUD[Org]):
    name = 'organization'
    model = Org

    _builder = CRUDQueries(
        base=t_organizations,
        id_fields=('org_id',),
        mapper_cls=OrgDataMapper,
        dumper_cls=OrgDataDumper,
    )

    async def get(self, org_id: int, *, for_update: bool = False) -> Org:  # type: ignore
        query, mapper = self._builder.select(id_values=(org_id,))
        if for_update:
            query = query.with_for_update(skip_locked=False, key_share=True)
        exception = self.model.DoesNotExist
        return mapper(await self._query_one(query, raise_=exception))

    async def check_exists(self, org_id: int) -> bool:
        query = (
            select([t_organizations]).
            where(t_organizations.c.org_id == org_id).
            limit(1)
        )
        return (await self._query_one(query) is not None)

    async def ensure_exists(self, org_id: int) -> None:
        query = (insert(t_organizations).
                 values(org_id=org_id).
                 on_conflict_do_nothing())
        await self.conn.execute(query)

    async def get_or_create(self, org_id: int, for_update: bool = False) -> Org:
        await self.ensure_exists(org_id)
        return await self.get(org_id, for_update=for_update)
