from typing import AsyncIterable, Callable, Mapping, Optional, Tuple

from sqlalchemy import and_, func, or_, tuple_

from sendr_aiopg.query_builder import CRUDQueries, Filters, RelationDescription, SelectableDataMapper

from mail.ipa.ipa.core.entities.collector import Collector
from mail.ipa.ipa.core.entities.user import User
from mail.ipa.ipa.storage.db.tables import collectors as t_collectors
from mail.ipa.ipa.storage.db.tables import users as t_users
from mail.ipa.ipa.storage.exceptions import UserAlreadyExists, UserNotFound
from mail.ipa.ipa.storage.mappers.base import BaseMapper
from mail.ipa.ipa.storage.mappers.collector.serialization import CollectorDataMapper
from mail.ipa.ipa.storage.mappers.user.serialization import UserDataDumper, UserDataMapper


class UserMapper(BaseMapper):
    name = 'user'

    _collector_relation = RelationDescription(
        name='collector',
        base=t_users,
        related=t_collectors,
        base_cols=('user_id',),
        related_cols=('user_id',),
        mapper_cls=CollectorDataMapper,
        outer_join=True,
    )
    _builder = CRUDQueries(
        base=t_users,
        id_fields=('user_id',),
        mapper_cls=UserDataMapper,
        dumper_cls=UserDataDumper,
        related=(_collector_relation,),
    )

    @staticmethod
    def _map_related(row: Mapping,
                     mapper: Callable[[Mapping], User],
                     rel_mappers: Mapping[str, SelectableDataMapper],
                     ) -> User:
        user = mapper(row)
        if 'collector' in rel_mappers:
            # checking mapped collector first, since it's outer join
            collector: Collector = rel_mappers['collector'](row)
            user.collector = collector if collector.user_id == user.user_id else None
        return user

    async def create(self, user: User) -> User:
        user.created_at = user.modified_at = func.now()
        query, mapper = self._builder.insert(user, ignore_fields=self._builder.id_fields)
        query = query.on_conflict_do_nothing()
        return mapper(await self._query_one(query, raise_=UserAlreadyExists))

    async def get(self, user_id: int) -> User:
        query, mapper = self._builder.select(id_values=(user_id,))
        return mapper(await self._query_one(query, raise_=UserNotFound))

    async def delete(self, user: User) -> None:
        query = self._builder.delete(user)
        await self._query_one(query, raise_=UserNotFound)

    async def get_or_create(self, user: User) -> User:
        try:
            return await self.create(user)
        except UserAlreadyExists:
            return await self.find_one(org_id=user.org_id, login=user.login)

    async def find(self,
                   org_id: Optional[int] = None,
                   login: Optional[str] = None,
                   has_error: Optional[bool] = None,
                   order_by: Optional[str] = None,
                   desc: bool = False,
                   limit: Optional[int] = None,
                   offset: Optional[int] = None,
                   ) -> AsyncIterable[User]:
        filters = Filters()
        filters.add_not_none('org_id', org_id)
        filters.add_not_none('error', has_error, lambda field: has_error != field.is_(None))
        filters.add_not_none('login', login)
        order: Optional[Tuple[str, ...]]
        if order_by is not None:
            order_by = '-' + order_by if desc else order_by
            order = (order_by,)
        else:
            order = None
        query, mapper = self._builder.select(filters=filters, order=order, limit=limit, offset=offset)
        async for row in self._query(query):
            yield mapper(row)

    async def find_one(self, org_id: int, login: str) -> User:
        users = [user async for user in self.find(org_id, login)]
        if len(users) == 0:
            raise UserNotFound
        assert len(users) == 1
        return users[0]

    async def save(self, user: User) -> User:
        user.modified_at = func.now()
        query, mapper = self._builder.update(user)
        return mapper(await self._query_one(query, raise_=UserNotFound))

    async def _get_batch_with_collectors(self,
                                         org_id: int,
                                         user_id: Optional[int] = None,
                                         collector_id: Optional[int] = None,
                                         batch_size: Optional[int] = None,
                                         only_errors: bool = False,
                                         ) -> AsyncIterable[User]:
        query, mapper, rel_mappers = self._builder.select_related(
            filters={'org_id': org_id},
            order=('user_id', 'collector.collector_id'),
            limit=batch_size,
        )
        if only_errors:
            query = query.where(
                or_(
                    t_users.c.error != None,  # noqa
                    and_(t_collectors.c.status != None, t_collectors.c.status != Collector.OK_STATUS),  # noqa
                )
            )
        if user_id is not None:
            query = query.where(
                tuple_(t_users.c.user_id, t_collectors.c.collector_id) > (user_id, collector_id)
            )
        async for row in self._query(query):
            yield self._map_related(row, mapper, rel_mappers)

    async def get_all_with_collectors(self,
                                      org_id: int,
                                      batch_size: int,
                                      only_errors: bool = False,
                                      ) -> AsyncIterable[User]:
        user_id = collector_id = None
        count = batch_size
        while count == batch_size:
            count = 0
            async for user in self._get_batch_with_collectors(
                org_id=org_id,
                user_id=user_id,
                collector_id=collector_id,
                batch_size=batch_size,
                only_errors=only_errors,
            ):
                count += 1
                user_id = user.user_id
                collector_id = user.collector.collector_id if isinstance(user.collector, Collector) else None
                yield user
