from datetime import datetime, timedelta, timezone

from sendr_taskqueue import BaseStorageArbiterWorker, BaseWorkerApplication
from sendr_utils import utcnow

from mail.ipa.ipa.conf import settings
from mail.ipa.ipa.core.entities.enums import TaskType, WorkerState
from mail.ipa.ipa.storage import Storage, StorageContext
from mail.ipa.ipa.utils.stats import collector_check_period_gauge, queue_size_gauge, queue_tasks_gauge


class ArbiterWorker(BaseStorageArbiterWorker):
    CHECK_WORKERS_ACTIVE = True
    KILL_ON_CLEANUP = True

    storage_context_cls = StorageContext
    worker_heartbeat_period = settings.TASKQ_WORKER_HEARTBEAT_PERIOD

    async def count_tasks(self, storage: Storage) -> None:
        queue_size: int = 0
        for type_ in TaskType:
            queue_tasks_gauge.labels(type_.value).observe(0)
        async for type_, count in storage.task.count_pending_by_type():
            queue_tasks_gauge.labels(type_.value).observe(count)
            queue_size += count
        queue_size_gauge.observe(queue_size)

        collector_min_checked_at = await storage.collector.get_min_checked_at()
        if not collector_min_checked_at:
            collector_min_checked_at = datetime(1970, 1, 1, tzinfo=timezone.utc)
        collector_check_period = (utcnow() - collector_min_checked_at).total_seconds()
        collector_check_period_gauge.observe(collector_check_period)

    async def clean_tasks(self, app: BaseWorkerApplication) -> None:
        async with self.storage_context(transact=True) as storage:
            await self.fail_workers(storage)
            await self.cleanup_workers(storage, (WorkerState.SHUTDOWN, WorkerState.FAILED))
            await self.count_tasks(storage)

        async with self.storage_context(transact=True) as storage:
            await storage.task.clean_old_tasks(timedelta(days=settings.TASKQ_FINISHED_TASK_MAX_AGE_DAYS))
            await storage.worker.clean_old_workers(timedelta(days=settings.WORKERS_SHUTDOWN_WORKER_MAX_AGE_DAYS))
