from asyncio import CancelledError
from datetime import timedelta
from typing import ClassVar

from sqlalchemy import func

from sendr_aiopg.storage.lock import LockAlreadyAcquired
from sendr_utils import copy_context

from mail.ipa.ipa.conf import settings
from mail.ipa.ipa.core.actions.collectors.check import CheckCollectorStatusAction
from mail.ipa.ipa.core.entities.enums import WorkerType
from mail.ipa.ipa.storage.exceptions import StorageNotFound
from mail.ipa.ipa.taskq.base import BaseWorker
from mail.ipa.ipa.utils.stats import queue_tasks_counter, queue_tasks_time


class CollectorWorker(BaseWorker):
    worker_type = WorkerType.COLLECTOR

    COLLECTOR_CHECK_DELAY: ClassVar[timedelta] = timedelta(minutes=settings.TASKQ_COLLECTOR_CHECK_DELAY_MINUTES)
    DB_TRANSACTION_TIMEOUT: ClassVar[timedelta] = timedelta(seconds=30)

    @copy_context
    async def process_task(self) -> bool:
        with self.logger, queue_tasks_time.labels(self.worker_type.value).time:
            async with self.storage_context(transact=True, transaction_timeout=self.DB_TRANSACTION_TIMEOUT) as storage:
                CheckCollectorStatusAction.context.storage = storage
                try:
                    collector = await storage.collector.get_for_work(self.COLLECTOR_CHECK_DELAY)
                except (StorageNotFound, LockAlreadyAcquired):
                    return self.PROCESS_TASK_WITH_PAUSE
                except CancelledError:
                    raise
                except Exception:
                    self.logger.exception('Failed to get task')
                    return self.PROCESS_TASK_WITH_PAUSE

                self.logger.context_push(collector_id=collector.collector_id)

                try:
                    collector = await CheckCollectorStatusAction(collector).run()
                except CancelledError:
                    raise
                except Exception:
                    self.logger.exception('Failed to run action for task')
                    return self.PROCESS_TASK_WITH_PAUSE
                finally:
                    collector.checked_at = func.now()
                    await storage.collector.save(collector, update_modified_at=False)
                    queue_tasks_counter.labels('check').inc()

                return self.PROCESS_TASK_WITH_NO_PAUSE
