from typing import ClassVar, Tuple, Type

from mail.ipa.ipa.conf import settings
from mail.ipa.ipa.core.actions.base import BaseAction
from mail.ipa.ipa.core.actions.collectors.edit import EditCollectorAction
from mail.ipa.ipa.core.actions.collectors.set_enabled import SetCollectorEnabledAction
from mail.ipa.ipa.core.actions.import_.user import InitUserImportAction
from mail.ipa.ipa.core.actions.users.remove_collectors import RemoveUserCollectorsAction
from mail.ipa.ipa.core.entities.enums import TaskState, TaskType, WorkerType
from mail.ipa.ipa.core.entities.task import Task
from mail.ipa.ipa.core.exceptions import (
    InvalidBirthdayError, NotConnectUIDError, PasswordSymbolForbiddenError, PasswordWeakError, SrcLoginEmptyError,
    UnsupportedGenderError, UnsupportedLanguageError
)
from mail.ipa.ipa.interactions.directory.exceptions import (
    DirectoryBirthdayInvalidError, DirectoryDepartmentExistsError, DirectoryLoginLongError,
    DirectoryLoginProhibitedError, DirectoryPasswordShortError, DirectoryPasswordWeakError
)
from mail.ipa.ipa.interactions.yarm.exceptions import YarmCollectFromHimselfError
from mail.ipa.ipa.storage import Storage
from mail.ipa.ipa.storage.mappers.task import TaskMapper
from mail.ipa.ipa.taskq.base import BaseWorker


class UserWorker(BaseWorker):
    worker_type = WorkerType.USER

    SAME_METATASK_PROCESSING_LIMIT: ClassVar[int] = settings.TASKQ_SAME_METATASK_PROCESSING_LIMIT

    task_action_mapping = {
        TaskType.EDIT_COLLECTOR: EditCollectorAction,
        TaskType.INIT_USER_IMPORT: InitUserImportAction,
        TaskType.SET_COLLECTOR_ENABLED: SetCollectorEnabledAction,
        TaskType.REMOVE_USER_COLLECTORS: RemoveUserCollectorsAction,
    }

    PERSISTENT_EXCEPTIONS: ClassVar[Tuple[Type[Exception], ...]] = (
        DirectoryBirthdayInvalidError,
        DirectoryDepartmentExistsError,
        DirectoryLoginLongError,
        DirectoryLoginProhibitedError,
        DirectoryPasswordShortError,
        DirectoryPasswordWeakError,
        InvalidBirthdayError,
        NotConnectUIDError,
        PasswordSymbolForbiddenError,
        PasswordWeakError,
        SrcLoginEmptyError,
        UnsupportedGenderError,
        UnsupportedLanguageError,
        YarmCollectFromHimselfError,
    )

    async def fetch_task_for_work(self, storage: Storage) -> Task:
        task_mapper: TaskMapper = storage[self.mapper_name_task]

        task = await task_mapper.get_user_task_for_work(
            task_types=self.task_types,
            same_metatask_processing_limit=self.SAME_METATASK_PROCESSING_LIMIT,
        )
        return task

    async def commit_task(self, task: Task, storage: Storage) -> None:
        if task.state not in (TaskState.PENDING, TaskState.PROCESSING):
            if 'user_info' in task.params:
                task.params['user_info'] = None
            if 'password' in task.params:
                task.params['password'] = None
            if 'users' in task.params:
                for user in task.params['users']:
                    if 'password' in user:
                        user['password'] = None

        return await super().commit_task(task, storage)

    def should_retry_exception(self, action_cls: Type[BaseAction], action_exception: Exception) -> bool:
        return not isinstance(action_exception, self.PERSISTENT_EXCEPTIONS)
