CREATE TABLE ipa.organizations (
    org_id BIGINT PRIMARY KEY
);

CREATE TABLE ipa.users (
    user_id BIGSERIAL PRIMARY KEY,
    uid BIGINT,
    suid BIGINT,
    org_id BIGINT NOT NULL,
    login TEXT NOT NULL,
    created_at timestamptz DEFAULT NOW() NOT NULL,
    modified_at timestamptz DEFAULT NOW() NOT NULL,
    error TEXT DEFAULT NULL,
    CONSTRAINT users_org_id_login_key UNIQUE (org_id, login)
);

CREATE INDEX users_modified_at_idx on ipa.users (modified_at);

CREATE TABLE ipa.collectors (
    collector_id BIGSERIAL PRIMARY KEY,
    user_id BIGINT REFERENCES ipa.users(user_id),
    pop_id TEXT,
    enabled BOOLEAN NOT NULL,

    params JSONB NOT NULL,

    total INTEGER,
    collected INTEGER,
    errors INTEGER,
    status TEXT,
    checked_at TIMESTAMPTZ DEFAULT NOW() NOT NULL,
    modified_at TIMESTAMPTZ DEFAULT NOW() NOT NULL,
    created_at TIMESTAMPTZ DEFAULT NOW() NOT NULL
);

CREATE INDEX collectors_checked_at_idx ON ipa.collectors (checked_at);
CREATE INDEX collectors_modified_at_idx on ipa.collectors (modified_at);
CREATE INDEX collectors_user_id_idx ON ipa.collectors (user_id);

CREATE TYPE ipa.task_state as enum ('failed', 'pending', 'processing', 'finished', 'deleted', 'cleanup');
CREATE TYPE ipa.worker_state as enum ('running', 'shutdown', 'failed', 'cleanedup');
CREATE TYPE ipa.task_type as enum (
  'edit_collector',
  'init_import',
  'init_user_import',
  'parse_csv',
  'remove_user_collectors',
  'set_collector_enabled',
  'stop_import'
);
CREATE TYPE ipa.worker_type as enum ('organization', 'user', 'collector');

CREATE TABLE ipa.tasks
(
    task_id     BIGSERIAL                              NOT NULL
        CONSTRAINT tasks_pkey PRIMARY KEY,
    task_type   ipa.task_type                 NOT NULL,
    entity_id   BIGINT,
    meta_task_id   BIGINT CONSTRAINT tasks_meta_task_id_fkey REFERENCES ipa.tasks,
    meta_info   JSONB,
    state       ipa.task_state                NOT NULL,
    params      JSONB,
    details     JSONB,
    retries     INTEGER                  DEFAULT 0     NOT NULL,
    run_at      TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
    created     TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
    updated     TIMESTAMP WITH TIME ZONE DEFAULT NOW() NOT NULL,
    action_name TEXT
);

CREATE INDEX tasks_state_type_run_at_idx ON ipa.tasks (state, task_type, run_at);
CREATE INDEX tasks_created_task_id_idx ON ipa.tasks (entity_id, created, task_id);
CREATE INDEX tasks_meta_task_id_idx ON ipa.tasks (meta_task_id) WHERE state IN ('pending', 'processing');

--Want to improve this? Add "where task_type in ('parse_csv', 'init_import')".
CREATE INDEX tasks_entity_id_created_task_id_idx ON ipa.tasks (entity_id, created, task_id);


CREATE TABLE ipa.workers
(
    worker_id   TEXT                      NOT NULL
        CONSTRAINT workers_pkey PRIMARY KEY,
    worker_type ipa.worker_type  NOT NULL,
    host        TEXT                      NOT NULL,
    state       ipa.worker_state NOT NULL,
    heartbeat   TIMESTAMP WITH TIME ZONE,
    startup     TIMESTAMP WITH TIME ZONE,
    task_id     BIGINT
        CONSTRAINT workers_task_id_fkey REFERENCES ipa.tasks
);
