package ru.yandex.scheduler;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.StreamEx;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.commune.alive2.AliveAppInfo;
import ru.yandex.commune.alive2.AliveAppsHolder;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.bazinga.scheduler.CronTaskInfo;
import ru.yandex.commune.bazinga.scheduler.OnetimeTaskInfo;
import ru.yandex.commune.bazinga.scheduler.RandomWorkerChooser;
import ru.yandex.commune.bazinga.scheduler.WorkerChooser;
import ru.yandex.commune.bazinga.scheduler.WorkerMeta;

@Slf4j
public class QuorumWorkerChooser implements WorkerChooser {
    private final AliveAppsHolder holder;
    private final RandomWorkerChooser chooser = RandomWorkerChooser.INSTANCE;

    public QuorumWorkerChooser(AliveAppsHolder holder) {
        this.holder = holder;
    }

    @Override
    public <T extends WorkerMeta> ListF<T> choose(CronTaskInfo cronTask, ListF<T> workers) {
        return chooser.choose(cronTask, choose(holder.aliveApps(), workers));
    }

    @Override
    public <T extends WorkerMeta> ListF<T> choose(OnetimeTaskInfo taskInfo, OnetimeJob job, ListF<T> workers) {
        return chooser.choose(taskInfo, job, choose(holder.aliveApps(), workers));
    }

    private static String getWorkerHost(WorkerMeta worker) {
        return worker.getHostPort().getHost();
    }

    private static <T extends WorkerMeta> ListF<T> choose(ListF<AliveAppInfo> aliveApps, ListF<T> workers) {
        val workerAliveApps = StreamEx.of(aliveApps).filter(app -> app.getAppName().contains("worker")).toImmutableList();
        val aliveAppsHosts = StreamEx.of(workerAliveApps).map(AliveAppInfo::getHostname).toImmutableSet();

        if (aliveAppsHosts.isEmpty()) {
            log.warn("Currently no workers are found amongst alive apps. Using default randomized selection for all workers");
            return workers;
        }

        log.debug("Alive apps info: {}", getVersionDescription((workerAliveApps)));

        val workerHosts = StreamEx.of(workers).map(QuorumWorkerChooser::getWorkerHost).toImmutableSet();
        log.debug("Available workers info: {}", workerHosts);

        warnAboutSeveralVersionsAvailable(workerAliveApps);
        warnAboutMissingHosts(aliveAppsHosts, workerHosts);

        val matchedWorkerHosts = findMatchingHosts(aliveAppsHosts, workerHosts);

        if (matchedWorkerHosts.isEmpty()) {
            log.warn("Failed to find any worker with host specified in alive apps. Using default randomized selection for all workers");
            return workers;
        }

        val version = getMostPopularVersion(workerAliveApps);
        log.debug("Selected as most popular version {}", version);
        val versionHosts = StreamEx.of(workerAliveApps)
                .filter(app -> version.equals(app.getVersion()))
                .map(AliveAppInfo::getHostname).toImmutableSet();
        log.debug("The workers matching the version are from following hosts: {}", versionHosts);

        return workers.filter(worker -> versionHosts.contains(getWorkerHost(worker)));
    }

    private static String getVersionDescription(List<AliveAppInfo> aliveAppInfos) {
        return StreamEx.of(aliveAppInfos)
                .sortedBy(AliveAppInfo::getVersion)
                .mapToEntry(AliveAppInfo::getVersion, AliveAppInfo::getHostname)
                .collapseKeys()
                .mapValues(hosts -> String.join(", ", hosts))
                .join(": ")
                .joining("; ");
    }

    private static String getMostPopularVersion(List<AliveAppInfo> aliveAppInfos) {
        return StreamEx.of(aliveAppInfos)
                .map(AliveAppInfo::getVersion)
                .sorted()
                .mapToEntry(Function.identity(), Function.identity())
                .collapseKeys()
                .mapValues(Collection::size)
                .sorted(QuorumWorkerChooser::compare)
                .findFirst()
                .map(Map.Entry::getKey)
                .orElseThrow(() -> new IllegalStateException("At least one alive app should be present in the list"));
    }

    private static int compare(Map.Entry<String, Integer> a, Map.Entry<String, Integer> b) {
        // Сортируем в порядке убывания - сначала числа воркеров, потом версии
        val countA = a.getValue();
        val countB = b.getValue();
        if (countA != countB) {
            return countB.compareTo(countA);
        }

        val versionA = a.getKey();
        val versionB = b.getKey();
        return versionB.compareTo(versionA);
    }

    private static void warnAboutSeveralVersionsAvailable(List<AliveAppInfo> aliveAppInfos) {
        if (aliveAppInfos.stream().map(AliveAppInfo::getVersion).distinct().count() > 1) {
            log.warn("There are several versions of workers currently available. This situation should only happen during deploy. Versions are: {}",
                    getVersionDescription(aliveAppInfos));
        }
    }

    private static Set<String> findMatchingHosts(Set<String> aliveHolderHosts, Set<String> workerHosts) {
        return StreamEx.of(workerHosts).filter(aliveHolderHosts::contains).toImmutableSet();
    }

    private static void warnAboutMissingHosts(Set<String> aliveHolderHosts, Set<String> workerHosts) {
        val missingHosts = StreamEx.of(workerHosts).filter(x -> !aliveHolderHosts.contains(x)).toImmutableSet();
        if (!missingHosts.isEmpty()) {
            log.warn("The following worker hosts are missing: {}", missingHosts);
        }
    }
}
