package ru.yandex.mail.diffusion;

import lombok.experimental.UtilityClass;
import lombok.val;
import one.util.streamex.StreamEx;
import ru.yandex.mail.diffusion.patch.FieldChange;
import ru.yandex.mail.diffusion.patch.exception.InconsistentPatchException;
import ru.yandex.mail.diffusion.patch.exception.InvalidValueException;

import java.util.HashSet;
import java.util.Set;

import static java.util.function.Predicate.not;

@UtilityClass
public class ApplierOps {
    @SuppressWarnings("unchecked")
    public static <T> Set<T> applySetChange(String fieldName, Set<T> fieldValue, FieldChange change, Class elementType) {
        Set<T> added = change.getAdded(elementType);
        Set<T> removed = change.getRemoved(elementType);

        val result = new HashSet<>(fieldValue);
        result.removeAll(removed);

        if (fieldValue.size() - result.size() != removed.size()) {
            val inconsistentElements = StreamEx.of(removed)
                .filter(not(fieldValue::contains))
                .joining("\n");
            throw new InconsistentPatchException("Set field '" + fieldName +
                "' does not contain the elements that patch is trying to remove:\n" + inconsistentElements);
        }

        result.addAll(added);
        return result;
    }

    private static long getPrimitive(String fieldName, FieldChange change, long minValue, long maxValue) {
        val value = change.getLong();
        if (value > maxValue || value < minValue) {
            throw new InvalidValueException(fieldName, "Integer '" + value +
                "' out of bounds [" + minValue + ", " + maxValue + ']');
        }
        return value;
    }

    public static byte byteValue(String fieldName, FieldChange change) {
        return (byte) getPrimitive(fieldName, change, Byte.MIN_VALUE, Byte.MAX_VALUE);
    }

    public static Byte boxedByteValue(String fieldName, FieldChange change) {
        return byteValue(fieldName, change);
    }

    public static short shortValue(String fieldName, FieldChange change) {
        return (short) getPrimitive(fieldName, change, Short.MIN_VALUE, Short.MAX_VALUE);
    }

    public static Short boxedShortValue(String fieldName, FieldChange change) {
        return shortValue(fieldName, change);
    }

    public static int intValue(String fieldName, FieldChange change) {
        return (int) getPrimitive(fieldName, change, Integer.MIN_VALUE, Integer.MAX_VALUE);
    }

    public static Integer boxedIntegerValue(String fieldName, FieldChange change) {
        return intValue(fieldName, change);
    }

    public static long longValue(String fieldName, FieldChange change) {
        return change.getLong();
    }

    public static Long boxedLongValue(String fieldName, FieldChange change) {
        return change.getLong();
    }
}
