package ru.yandex.mail.diffusion.generator;

import javassist.ClassClassPath;
import javassist.ClassPool;
import javassist.CtClass;
import lombok.SneakyThrows;
import lombok.val;
import one.util.streamex.StreamEx;
import ru.yandex.mail.diffusion.ApplierOps;
import ru.yandex.mail.diffusion.Diffusion;
import ru.yandex.mail.diffusion.FieldMatcher;
import ru.yandex.mail.diffusion.patch.FieldChange;
import ru.yandex.mail.diffusion.patch.PatchApplier;
import ru.yandex.mail.diffusion.patch.PatchProvider;

import java.io.BufferedOutputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;

public class CodeGenerator {
    private static final String[] DEFAULT_IMPORT = {
        "ru.yandex.mail.diffusion",
        "ru.yandex.mail.diffusion.patch",
        "ru.yandex.mail.diffusion.patch.exception"
    };

    private static final Class<?>[] DEFAULT_CLASSPATH = {
        Diffusion.class,
        FieldMatcher.class,
        PatchApplier.class,
        PatchProvider.class,
        FieldChange.class,
        ApplierOps.class
    };

    private final ClassPool pool;
    private final FieldMatcherGenerator fieldMatcherGenerator;
    private final PatchApplierGenerator patchApplierGenerator;
    private final MetaInfoCollector metaInfoCollector;

    @SneakyThrows
    private static void writeClassFile(CtClass clazz, String dir) {
        try (val out = new FileOutputStream(dir + File.separator + clazz.getSimpleName() + ".class");
             val outStream = new DataOutputStream(new BufferedOutputStream(out))) {
            clazz.toBytecode(outStream);
        }
    }

    @SneakyThrows
    @SuppressWarnings("unchecked")
    private static <T> T createProxy(CtClass cls, Class<?> processed) {
        val type = cls.toClass(processed.getClassLoader(), processed.getProtectionDomain());
        return (T) type.getDeclaredConstructor().newInstance();
    }

    public CodeGenerator() {
        pool = new ClassPool();

        StreamEx.of(DEFAULT_IMPORT).forEach(pool::importPackage);
        StreamEx.of(DEFAULT_CLASSPATH)
            .map(ClassClassPath::new)
            .forEach(pool::insertClassPath);

        fieldMatcherGenerator = new FieldMatcherGenerator(pool);
        patchApplierGenerator = new PatchApplierGenerator(pool);
        metaInfoCollector = new MetaInfoCollector();
    }

    public <T> Proxies<T> process(Class<T> type) {
        pool.insertClassPath(new ClassClassPath(type));
        val classInfo = metaInfoCollector.collect(type);

        val fieldMatcherClass = fieldMatcherGenerator.generate(classInfo);
        val patchApplierClass = patchApplierGenerator.generate(classInfo);

        return new Proxies<>(
            createProxy(fieldMatcherClass, type),
            createProxy(patchApplierClass, type)
        );
    }
}
