package ru.yandex.mail.diffusion.generator;

import javassist.ClassPool;
import javassist.CtClass;
import javassist.CtMethod;
import javassist.CtNewMethod;
import lombok.AllArgsConstructor;
import lombok.SneakyThrows;
import lombok.val;
import ru.yandex.mail.diffusion.FieldMatcher;

import static java.util.Map.entry;
import static ru.yandex.mail.diffusion.generator.Utils.format;

@AllArgsConstructor
class FieldMatcherGenerator {
    private final ClassPool pool;

    private static boolean isFloatingPoint(Class<?> type) {
        return type == float.class
            || type == Float.class
            || type == double.class
            || type == Double.class;
    }

    private static boolean isInteger(Class<?> type) {
        return type == byte.class
            || type == Byte.class
            || type == short.class
            || type == Short.class
            || type == int.class
            || type == Integer.class
            || type == long.class
            || type == Long.class;
    }

    private static String generateComparison(FieldInfo info) {
        val type = info.getType();

        String template;
        String cast = "";

        if (type.isPrimitive()) {
            if (isFloatingPoint(type)) {
                cast = "(double)";
            } else if (isInteger(type)) {
                cast = "(long)";
            }

            template = "if (oldObject.${getter}() != newObject.${getter}()) {"
                     + "    listener.onMismatch(\"${fieldName}\", ${cast} oldObject.${getter}(),"
                     + "                        ${cast} newObject.${getter}());"
                     + "}";
        } else {
            if (isFloatingPoint(type)) {
                cast = ".doubleValue()";
            } else if (isInteger(type)) {
                cast = ".longValue()";
            }

            template = "if (!oldObject.${getter}().equals(newObject.${getter}())) {"
                     + "    listener.onMismatch(\"${fieldName}\", oldObject.${getter}()${cast},"
                     + "                        newObject.${getter}()${cast});"
                     + "}";
        }

        return format(template,
            entry("getter", info.getGetterName()),
            entry("fieldName", info.getName()),
            entry("cast", cast)
        );
    }

    @SneakyThrows
    private static CtMethod generateMatch(ClassInfo info, CtClass declaring) {
        val comparisons = info.fieldsStream()
            .map(FieldMatcherGenerator::generateComparison)
            .joining();

        val source = format("public void match(${objectType} oldObject, ${objectType} newObject,"
                          + "                  FieldMatchListener listener) {"
                          +     comparisons
                          + "}",
            entry("objectType", info.getType().getCanonicalName())
        );
        return CtNewMethod.make(source, declaring);
    }

    @SneakyThrows
    private static CtMethod generateMatchBridge(ClassInfo info, CtClass declaring) {
        val source = format("public void match(Object oldObject, Object newObject, FieldMatchListener listener) {"
                          + "    match((${objectType}) oldObject, (${objectType}) newObject, listener);"
                          + "}",
            entry("objectType", info.getType().getCanonicalName())
        );
        return CtNewMethod.make(source, declaring);
    }

    @SneakyThrows
    CtClass generate(ClassInfo info) {
        val matcherClass = pool.makeClass(info.getFieldMatcherQualifiedName());

        val matcherInterfaceType = FieldMatcher.class.getCanonicalName();
        val signature = Utils.createGenericSignature(matcherInterfaceType, info.getType().getCanonicalName());
        matcherClass.addInterface(pool.get(matcherInterfaceType));
        matcherClass.setGenericSignature(signature.encode());

        matcherClass.addMethod(generateMatch(info, matcherClass));
        matcherClass.addMethod(generateMatchBridge(info, matcherClass));

        return matcherClass;
    }
}
